<?php
header('Content-Type: application/json; charset=UTF-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST');

class AudioProcessor {

    private $dev = '@OnyxApiTeam';
    private $uploadTemp;
    private $fileId;

    public function handle() {
        $input = $this->getInput();
        if (!$input) $this->fail('No valid file or URL provided');

        $this->fileId = $this->uploadToAudioAlter();
        if (!$this->fileId) $this->fail('Upload failed');

        sleep(3);
        if (!$this->applyEffect($this->fileId)) $this->fail('Applying 8D effect failed');

        $path = $this->downloadFinal($this->fileId);
        if (!$path) $this->fail('Download from AudioAlter failed');

        $url = $this->generateDownloadUrl($this->fileId);
        $this->respond(true, ['download' => $url]);
    }

    private function getInput() {
        $this->uploadTemp = sys_get_temp_dir() . '/audio_' . uniqid() . '.mp3';

        if (!empty($_REQUEST['url'])) {
            $url = rawurldecode($_REQUEST['url']);
            $url = $this->encodeUrl($url);
            if (!filter_var($url, FILTER_VALIDATE_URL)) return false;

            $data = @file_get_contents($url);
            if (!$data) return false;

            file_put_contents($this->uploadTemp, $data);
            return $this->uploadTemp;
        }

        if (!empty($_FILES['file']) && $_FILES['file']['error'] === UPLOAD_ERR_OK) {
            move_uploaded_file($_FILES['file']['tmp_name'], $this->uploadTemp);
            return $this->uploadTemp;
        }

        return false;
    }

    private function encodeUrl($url) {
        $parts = parse_url($url);
        if (!isset($parts['scheme']) || !isset($parts['host'])) return $url;

        $path = array_map('rawurlencode', explode('/', $parts['path']));
        $encodedPath = implode('/', $path);

        $query = '';
        if (isset($parts['query'])) {
            $query = '?' . str_replace(' ', '%20', $parts['query']);
        }

        $encodedUrl = $parts['scheme'] . '://' . $parts['host'] . $encodedPath . $query;
        if (isset($parts['fragment'])) {
            $encodedUrl .= '#' . rawurlencode($parts['fragment']);
        }

        return $encodedUrl;
    }

    private function uploadToAudioAlter() {
        $ch = curl_init("https://api.audioalter.com/pages/upload");
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => [
                "Accept: application/json, text/plain, */*",
                "Origin: https://audioalter.com",
                "Referer: https://audioalter.com/",
                "User-Agent: Mozilla/5.0"
            ],
            CURLOPT_POSTFIELDS => [
                "audio" => new CURLFile($this->uploadTemp),
                "preview" => "true"
            ]
        ]);

        $res = curl_exec($ch);
        curl_close($ch);
        @unlink($this->uploadTemp);

        $data = json_decode($res, true);
        return isset($data['files'][0]['id']) ? $data['files'][0]['id'] : null;
    }

    private function applyEffect($fileId) {
        $payload = json_encode([
            "id" => "8d-audio",
            "ids" => [$fileId],
            "effects" => [
                ["tool" => "autopanner", "params" => ["frequency" => 0.08, "depth" => 85]],
                ["tool" => "reverb", "params" => [
                    "reverberance" => 50, "preDelay" => 0, "damping" => 50,
                    "stereoDepth" => 100, "wet" => 0, "roomScale" => 100
                ]]
            ]
        ], JSON_UNESCAPED_SLASHES);

        $ch = curl_init("https://api.audioalter.com/pages/preset");
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => [
                "Content-Type: application/json",
                "Origin: https://audioalter.com",
                "Referer: https://audioalter.com/",
                "User-Agent: Mozilla/5.0"
            ],
            CURLOPT_POSTFIELDS => $payload
        ]);

        $res = curl_exec($ch);
        curl_close($ch);

        return !empty($res);
    }

    private function downloadFinal($fileId) {
        $url = "https://api.audioalter.com/download/" . $fileId;
        $filename = $fileId . ".mp3";
        $dir = __DIR__ . "/media";
        $path = $dir . "/" . $filename;

        if (!is_dir($dir)) mkdir($dir, 0755, true);

        $attempt = 0;
        while ($attempt++ < 10) {
            $data = @file_get_contents($url);
            if ($data && strlen($data) > 50000) {
                file_put_contents($path, $data);
                return $path;
            }
            sleep(2);
        }

        return null;
    }

    private function generateDownloadUrl($fileId) {
        $base = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off' ? "https" : "http") .
                "://" . $_SERVER['HTTP_HOST'] . dirname($_SERVER['PHP_SELF']);
        return $base . "/media/" . rawurlencode($fileId . ".mp3");
    }

    private function respond($ok, $data = []) {
        echo json_encode(array_merge([
            "ok" => $ok,
            "dev" => $this->dev
        ], $data), JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
        exit;
    }

    private function fail($msg) {
        $this->respond(false, ['error' => $msg]);
    }
}

$processor = new AudioProcessor();
$processor->handle();
