<?php

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');
if($_SERVER['REQUEST_METHOD'] === 'OPTIONS'){
    http_response_code(204);
    exit;
}

class DeepSeekAPI {
    private $pdo;
    private $apiKey = 'sk-aIEVEPNEUojnw0caBolbNJTz0WR9pzqkZjUzb4jNHX9MPUxi';
    private $endpoint = 'https://api.yescale.io/v1/chat/completions';
    private $model = 'deepseek-v3-0324';

    public function __construct(array $dbConfig) {
        $this->pdo = new PDO(
            "mysql:host={$dbConfig['host']};dbname={$dbConfig['dbname']};charset=utf8mb4",
            $dbConfig['user'],
            $dbConfig['pass'],
            [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
        );
        $this->createTables();
    }

    private function createTables() {
        $this->pdo->exec("CREATE TABLE IF NOT EXISTS chat_sessions (
            id VARCHAR(36) PRIMARY KEY,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )");
        $this->pdo->exec("CREATE TABLE IF NOT EXISTS chat_history (
            id INT AUTO_INCREMENT PRIMARY KEY,
            chat_id VARCHAR(36),
            role ENUM('user','assistant') NOT NULL,
            content TEXT NOT NULL,
            timestamp BIGINT NOT NULL,
            FOREIGN KEY (chat_id) REFERENCES chat_sessions(id) ON DELETE CASCADE
        )");
    }

    private function chatExists(string $chatId): bool {
        $stmt = $this->pdo->prepare("SELECT 1 FROM chat_sessions WHERE id = :id");
        $stmt->execute(['id' => $chatId]);
        return (bool)$stmt->fetchColumn();
    }

    private function generateUUIDv4(): string {
        $data = random_bytes(16);
        $data[6] = chr((ord($data[6]) & 0x0f) | 0x40);
        $data[8] = chr((ord($data[8]) & 0x3f) | 0x80);
        return vsprintf('%s%s-%s-%s-%s-%s%s%s', str_split(bin2hex($data), 4));
    }

    public function createChat(): array {
        do {
            $chatId = $this->generateUUIDv4();
        } while ($this->chatExists($chatId));
        $stmt = $this->pdo->prepare("INSERT INTO chat_sessions (id) VALUES (:id)");
        $stmt->execute(['id' => $chatId]);
        return ["ok" => true, "dev" => "@OnyxApiTeam", "result" => ["chatId" => $chatId]];
    }

    public function sendMessage(string $chatId, string $text): array {
        if (!$chatId || !$text) {
            http_response_code(400);
            return ["ok" => false, "error" => "Missing chat ID or message text."];
        }
        if (!$this->chatExists($chatId)) {
            http_response_code(403);
            return ["ok" => false, "error" => "Chat ID not found or invalid."];
        }

        $ts = round(microtime(true) * 1000);
        $stmt = $this->pdo->prepare("INSERT INTO chat_history (chat_id, role, content, timestamp) VALUES (?, 'user', ?, ?)");
        $stmt->execute([$chatId, $text, $ts]);

        $stmt = $this->pdo->prepare("SELECT role, content, timestamp FROM chat_history WHERE chat_id = ? ORDER BY timestamp DESC LIMIT 5");
        $stmt->execute([$chatId]);
        $history = array_reverse($stmt->fetchAll(PDO::FETCH_ASSOC));

        $messages = [];
        $messages[] = ["role" => "system", "content" => "You are DeepSeek AI, an intelligent and friendly assistant. Please respond helpfully, in detail, and with clear structure. Use markdown formatting when necessary."];
        foreach ($history as $msg) {
            $messages[] = ["role" => $msg['role'], "content" => $msg['content']];
        }

        $payload = [
            "model" => $this->model,
            "messages" => $messages,
            "max_tokens" => 1000,
            "temperature" => 0.7,
            "stream" => false
        ];

        $ch = curl_init($this->endpoint);
        curl_setopt_array($ch, [
            CURLOPT_HTTPHEADER => [
                "Authorization: Bearer {$this->apiKey}",
                "Content-Type: application/json"
            ],
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES)
        ]);
        $apiResponseRaw = curl_exec($ch);
        $curlErr = curl_error($ch);
        curl_close($ch);

        if ($curlErr) {
            http_response_code(502);
            return ["ok" => false, "error" => "Upstream connection failed."];
        }

        $apiResponse = json_decode($apiResponseRaw, true);
        if (!is_array($apiResponse) || empty($apiResponse['choices'][0]['message']['content'])) {
            http_response_code(500);
            return ["ok" => false, "error" => "Invalid response from upstream."];
        }

        $reply = $apiResponse['choices'][0]['message']['content'];
        $stmt = $this->pdo->prepare("INSERT INTO chat_history (chat_id, role, content, timestamp) VALUES (?, 'assistant', ?, ?)");
        $stmt->execute([$chatId, $reply, round(microtime(true) * 1000)]);
        return ["ok" => true, "dev" => "@Cactus_Dev", "result" => ["assistantReply" => $reply]];
    }

    public function getHistory(string $chatId): array {
        if (!$chatId) {
            http_response_code(400);
            return ["ok" => false, "error" => "Chat ID is missing."];
        }
        if (!$this->chatExists($chatId)) {
            http_response_code(403);
            return ["ok" => false, "error" => "Chat ID is invalid or not found."];
        }
        $stmt = $this->pdo->prepare("SELECT role, content, timestamp FROM chat_history WHERE chat_id = ? ORDER BY timestamp ASC");
        $stmt->execute([$chatId]);
        $messages = $stmt->fetchAll(PDO::FETCH_ASSOC);
        return ["ok" => true, "dev" => "@OnyxApiTeam", "result" => $messages];
    }

    public function deleteChat(string $chatId): array {
        if (!$chatId) {
            http_response_code(400);
            return ["ok" => false, "error" => "Chat ID to delete is missing."];
        }
        if (!$this->chatExists($chatId)) {
            http_response_code(403);
            return ["ok" => false, "error" => "Chat ID is invalid or doesn't exist."];
        }
        $this->pdo->prepare("DELETE FROM chat_history WHERE chat_id = ?")->execute([$chatId]);
        $this->pdo->prepare("DELETE FROM chat_sessions WHERE id = ?")->execute([$chatId]);
        return ["ok" => true, "dev" => "@OnyxApiTeam", "result" => "Chat deleted successfully."];
    }

    public static function input(string $key): ?string {
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $data = $_POST;
            if (empty($data)) {
                $raw = file_get_contents('php://input');
                $json = json_decode($raw, true);
                if (json_last_error() === JSON_ERROR_NONE && isset($json[$key])) return $json[$key];
            }
            return $data[$key] ?? null;
        } else {
            return $_GET[$key] ?? null;
        }
    }
}

$dbConfig = [
    'host' => 'localhost',
    'dbname' => 'cactusde_deepseekr1',
    'user' => 'cactusde_deepseekr1',
    'pass' => 'www09148215271s',
];

$api = new DeepSeekAPI($dbConfig);

$action = DeepSeekAPI::input('action');
$id = DeepSeekAPI::input('id');
$text = DeepSeekAPI::input('text');

switch ($action) {
    case 'create':
        $response = $api->createChat();
        break;
    case 'send':
        $response = $api->sendMessage($id, $text);
        break;
    case 'history':
        $response = $api->getHistory($id);
        break;
    case 'delete':
        $response = $api->deleteChat($id);
        break;
    default:
        http_response_code(400);
        $response = ["ok" => false, "error" => "Invalid request. Please send one of these actions: create, send, history, delete."];
}

echo json_encode($response, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
