<?php

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

$dev = '@Cactus_Dev';

$symbolMap = [
    'btc' => 'bitcoin', 'eth' => 'ethereum', 'usdt' => 'tether', 'xrp' => 'xrp', 'bnb' => 'bnb',
    'sol' => 'solana', 'usdc' => 'usd-coin', 'trx' => 'tron', 'doge' => 'dogecoin', 'ada' => 'cardano',
    'sui' => 'sui', 'busd' => 'usd-coin', 'bch' => 'bitcoin-cash', 'link' => 'chainlink',
    'leo' => 'unus-sed-leo', 'xlm' => 'stellar', 'avax' => 'avalanche', 'ton' => 'toncoin',
    'shib' => 'shiba-inu', 'ltc' => 'litecoin', 'hbar' => 'hedera', 'xmr' => 'monero',
    'ethena' => 'ethena', 'dot' => 'polkadot-new', 'dai' => 'multi-collateral-dai',
    'bgb' => 'bitget-token-new', 'uni' => 'uniswap', 'pepe' => 'pepe', 'pi' => 'pi',
    'aave' => 'aave', 'tao' => 'bittensor', 'okb' => 'okb', 'apt' => 'aptos',
    'icp' => 'internet-computer', 'cro' => 'cronos', 'near' => 'near-protocol', 'etc' => 'ethereum-classic',
    'ondo' => 'ondo-finance', 'gt' => 'gatetoken', 'mnt' => 'mantle', 'kas' => 'kaspa',
    'matic' => 'polygon-ecosystem-token', 'vet' => 'vechain', 'sky' => 'sky', 'rndr' => 'render',
    'atom' => 'cosmos', 'algo' => 'algorand', 'wdc' => 'worldcoin-org', 'fil' => 'filecoin',
    'arb' => 'arbitrum', 'fdusd' => 'first-digital-usd', 'kcs' => 'kucoin-token', 'sei' => 'sei',
    'kaia' => 'kaia', 'qnt' => 'quant', 'jup' => 'jupiter-ag', 'flare' => 'flare',
    'spx' => 'spx6900', 'bonk' => 'bonk1', 'vp' => 'virtual-protocol', 'inj' => 'injective',
    'tia' => 'celestia', 'bnx' => 'binaryx-new', 'fart' => 'fartcoin', 'stx' => 'stacks',
    'op' => 'optimism-ethereum', 'xdc' => 'xdc-network', 'story' => 'story-protocol',
    'pyusd' => 'paypal-usd', 'sonic' => 'sonic', 'paxg' => 'pax-gold', 'wif' => 'dogwifhat',
    'xaut' => 'tether-gold', 'vlta' => 'vaulta', 'grt' => 'the-graph', 'crv' => 'curve-dao-token',
    'imx' => 'immutable', 'nexo' => 'nexo', 'jito' => 'jito', 'aero' => 'aerodrome-finance',
    'cake' => 'pancakeswap', 'floki' => 'floki-inu', 'zec' => 'zcash', 'ens' => 'ethereum-name-service',
    'dexe' => 'dexe', 'theta' => 'theta-network', 'newton' => 'newton', 'sand' => 'sandbox',
    'lido' => 'lido-dao', 'gala' => 'gala', 'ppg' => 'pudgy-penguins', 'iota' => 'iota',
    'bsv' => 'bitcoin-sv', 'zro' => 'layerzero', 'btt' => 'bittorrent-new',
    'irt' => 'ir-toman', 'usd' => 'united-states-dollar', 'aud' => 'australian-dollar',
    'brl' => 'brazilian-real', 'cad' => 'canadian-dollar', 'chf' => 'swiss-franc',
    'clp' => 'chilean-peso', 'cny' => 'chinese-yuan', 'czk' => 'czech-koruna', 'dkk' => 'danish-krone',
    'eur' => 'euro', 'gbp' => 'pound-sterling', 'hkd' => 'hong-kong-dollar', 'huf' => 'hungarian-forint',
    'idr' => 'indonesian-rupiah', 'inr' => 'indian-rupee', 'jpy' => 'japanese-yen',
    'krw' => 'south-korean-won', 'mxn' => 'mexican-peso', 'myr' => 'malaysian-ringgit',
    'nok' => 'norwegian-krone', 'nzd' => 'new-zealand-dollar', 'php' => 'philippine-peso',
    'pkr' => 'pakistani-rupee', 'pln' => 'polish-zloty', 'rub' => 'russian-ruble',
    'sek' => 'swedish-krona', 'sgd' => 'singapore-dollar', 'thb' => 'thai-baht',
    'try' => 'turkish-lira', 'twd' => 'new-taiwan-dollar', 'zar' => 'south-african-rand',
    'aed' => 'united-arab-emirates-dirham', 'bgn' => 'bulgarian-lev', 'hrk' => 'croatian-kuna',
    'mur' => 'mauritian-rupee', 'ron' => 'romanian-leu', 'ngn' => 'nigerian-naira',
    'cop' => 'colombian-peso', 'ars' => 'argentine-peso', 'pen' => 'peruvian-sol',
    'vnd' => 'vietnamese-dong', 'uah' => 'ukrainian-hryvnia', 'bob' => 'bolivian-boliviano',
    'all' => 'albanian-lek', 'amd' => 'armenian-dram', 'azn' => 'azerbaijani-manat',
    'bam' => 'bosnia-herzegovina-convertible-mark', 'bdt' => 'bangladeshi-taka',
    'bhd' => 'bahraini-dinar', 'bmd' => 'bermudan-dollar', 'byr' => 'belarusian-ruble',
    'crc' => 'costa-rican-colon', 'cup' => 'cuban-peso', 'dop' => 'dominican-peso',
    'dzd' => 'algerian-dinar', 'egp' => 'egyptian-pound', 'gel' => 'georgian-lari',
    'ghs' => 'ghanaian-cedi', 'gtq' => 'guatemalan-quetzal', 'hnl' => 'honduran-lempir',
    'iqd' => 'iraqi-dinar', 'jmd' => 'jamaican-dollar', 'jod' => 'jordanian-dinar',
    'kes' => 'kenyan-shilling', 'kgs' => 'kyrgystani-som', 'khr' => 'cambodian-riel',
    'kwd' => 'kuwaiti-dinar', 'kzt' => 'kazakhstani-tenge', 'lbp' => 'lebanese-pound',
    'lkr' => 'sri-lankan-rupee', 'mad' => 'moroccan-dirham', 'mdl' => 'moldovan-leu',
    'mkd' => 'macedonian-denar', 'mmk' => 'myanma-kyat', 'mnt' => 'mongolian-tugrik',
    'nad' => 'namibian-dollar', 'nio' => 'nicaraguan-cordoba', 'npr' => 'nepalese-rupee',
    'omr' => 'omani-rial', 'pab' => 'panamanian-balboa', 'qar' => 'qatari-rial',
    'rsd' => 'serbian-dinar', 'sar' => 'saudi-riyal', 'ssp' => 'south-sudanese-pound',
    'tnd' => 'tunisian-dinar', 'ttd' => 'trinidad-and-tobago-dollar', 'ugx' => 'ugandan-shilling',
    'uyu' => 'uruguayan-peso', 'uzs' => 'uzbekistan-som', 'afn' => 'afghan-afghani',
];

function normalize($symbol, $map) {
    $key = strtolower(str_replace(['-', '_'], '', $symbol));
    return $map[$key] ?? $symbol;
}

$jsonBody = [];
if ($_SERVER['REQUEST_METHOD'] === 'POST' && strpos($_SERVER['CONTENT_TYPE'] ?? '', 'application/json') !== false) {
    $raw = file_get_contents('php://input');
    $jsonBody = json_decode($raw, true) ?: [];
}

function getParam($name) {
    global $jsonBody;
    return $_GET[$name] ?? $_POST[$name] ?? $jsonBody[$name] ?? null;
}

if (getParam('list') === 'true') {
    $digital = [];
    $fiat = [];

    foreach ($symbolMap as $short => $full) {
        if (preg_match('/(dollar|rial|toman|peso|franc|rupee|euro|lira|krone|won|yen|dirham|pound|lev|manat|dram|afghani|som|krona|zloty|tugrik|mark|colon|dinar)/i', $full)) {
            $fiat[$short] = $full;
        } else {
            $digital[$short] = $full;
        }
    }

    echo json_encode([
        'ok' => true,
        'dev' => $dev,
        'supported' => [
            'digital' => $digital,
            'fiat' => $fiat
        ]
    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
    exit;
}

$base = getParam('base') ? normalize(getParam('base'), $symbolMap) : null;
$quote = getParam('quote') ? normalize(getParam('quote'), $symbolMap) : null;
$amount = getParam('amount') ? trim(getParam('amount')) : null;

if (!$base || !$quote || !$amount || !is_numeric($amount) || $amount <= 0) {
    http_response_code(400);
    echo json_encode([
        'ok' => false,
        'dev' => $dev,
        'error' => 'Missing or invalid base, quote or amount parameter'
    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
    exit;
}

function nullIfEmpty($val) {
    return (isset($val) && $val !== '') ? $val : null;
}

function persianToEnglishNumbers(string $input): string {
    $persianNums = ['۰','۱','۲','۳','۴','۵','۶','۷','۸','۹', '٫', '٬'];
    $englishNums = ['0','1','2','3','4','5','6','7','8','9', '.', ','];
    return str_replace($persianNums, $englishNums, $input);
}

function convertNumbersInArray($arr) {
    foreach ($arr as $key => &$val) {
        if (is_array($val)) {
            $val = convertNumbersInArray($val);
        } elseif (is_string($val)) {
            $val = persianToEnglishNumbers($val);
        }
    }
    return $arr;
}

$url = "https://arzdigital.com/convert/?b={$base}&q={$quote}&amount={$amount}";

function getTextFromXPath($xpath, $query) {
    $node = $xpath->query($query);
    if ($node && $node->length > 0) {
        return trim($node->item(0)->nodeValue);
    }
    return null;
}

function getAttributeFromXPath($xpath, $query, $attribute) {
    $node = $xpath->query($query);
    if ($node && $node->length > 0) {
        return trim($node->item(0)->getAttribute($attribute));
    }
    return null;
}

function getHtml($url) {
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64)');
    $html = curl_exec($ch);
    curl_close($ch);
    return $html;
}

$html = getHtml($url);
if (!$html) {
    http_response_code(500);
    echo json_encode(['ok' => false, 'dev' => $dev, 'error' => 'Failed to fetch HTML'], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

libxml_use_internal_errors(true);
$dom = new DOMDocument();
$dom->loadHTML(mb_convert_encoding($html, 'HTML-ENTITIES', 'UTF-8'));
libxml_clear_errors();

$xpath = new DOMXPath($dom);

$base_amount = getTextFromXPath($xpath, "//span[contains(@class, 'baseval-autonumber')]") ?: $amount;
$base_name = getTextFromXPath($xpath, "//span[@data-av-value='faname-base']") ?: $base;
$quote_amount = getTextFromXPath($xpath, "//span[contains(@class, 'quoteval-autonumber')]");
$quote_name = getTextFromXPath($xpath, "//span[@data-av-value='faname-quote']") ?: $quote;
$total_value_toman = getTextFromXPath($xpath, "//span[@data-av-value='total-value-irt']");
$conversion_rate = getTextFromXPath($xpath, "//span[@data-av-value='1baseAsQuote']");

$base_price_usd = getTextFromXPath($xpath, "//div[contains(@class,'basetodollar')]//div[@data-av-value='1basePriceDollar']");
$base_change_usd = getTextFromXPath($xpath, "//div[contains(@class,'basetodollar')]//div[@data-av-value='1baseChangeDollar']");
$base_price_irt = getTextFromXPath($xpath, "//div[contains(@class,'basetotoman')]//div[@data-av-value='1basePriceIrt']");
$base_change_irt = getTextFromXPath($xpath, "//div[contains(@class,'basetotoman')]//div[@data-av-value='1baseChangeIrt']");
$quote_price_usd = getTextFromXPath($xpath, "//div[contains(@class,'quotetodollar')]//div[@data-av-value='1quotePriceDollar']");
$quote_change_usd = getTextFromXPath($xpath, "//div[contains(@class,'quotetodollar')]//div[@data-av-value='1quoteChangeDollar']");
$quote_price_irt = getTextFromXPath($xpath, "//div[contains(@class,'quotetotoman')]//div[@data-av-value='1quotePriceIrt']");
$quote_change_irt = getTextFromXPath($xpath, "//div[contains(@class,'quotetotoman')]//div[@data-av-value='1quoteChangeIrt']");

$base_icon = getAttributeFromXPath($xpath, "//img[@data-av-value='logo-base']", "src");
$quote_icon = getAttributeFromXPath($xpath, "//img[@data-av-value='logo-quote']", "src");

$base_invalid = is_null($base_icon) && is_null($base_price_usd) && is_null($base_price_irt);
$quote_invalid = is_null($quote_icon) && is_null($quote_price_usd) && is_null($quote_price_irt);

if ($base_invalid || $quote_invalid) {
    http_response_code(404);
    echo json_encode([
        'ok' => false,
        'dev' => $dev,
        'error' => 'Base or quote currency not found or not supported'
    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
    exit;
}

$response = [
    'ok' => true,
    'dev' => $dev,
    'data' => [
        'base' => [
            'amount' => nullIfEmpty($base_amount),
            'symbol' => nullIfEmpty($base_name),
            'image_url' => nullIfEmpty($base_icon),
            'price' => [
                'usd' => [
                    'price' => nullIfEmpty($base_price_usd),
                    'change_percent' => nullIfEmpty($base_change_usd),
                ],
                'toman' => [
                    'price' => nullIfEmpty($base_price_irt),
                    'change_percent' => nullIfEmpty($base_change_irt),
                ],
            ],
        ],
        'quote' => [
            'amount' => nullIfEmpty($quote_amount),
            'symbol' => nullIfEmpty($quote_name),
            'image_url' => nullIfEmpty($quote_icon),
            'price' => [
                'usd' => [
                    'price' => nullIfEmpty($quote_price_usd),
                    'change_percent' => nullIfEmpty($quote_change_usd),
                ],
                'toman' => [
                    'price' => nullIfEmpty($quote_price_irt),
                    'change_percent' => nullIfEmpty($quote_change_irt),
                ],
            ],
        ],
        'total_value_toman' => nullIfEmpty($total_value_toman),
        'conversion_rate' => nullIfEmpty($conversion_rate),
    ],
];

$response = convertNumbersInArray($response);
echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);

