<?php


if (!function_exists('mb_strtolower')) {
    function mb_strtolower(string $string, ?string $encoding = null): string
    {
        return strtolower($string);
    }
}

if (!function_exists('mb_strtoupper')) {
    function mb_strtoupper(string $string, ?string $encoding = null): string
    {
        return strtoupper($string);
    }
}

if (!function_exists('mb_convert_encoding')) {
    function mb_convert_encoding(string $string, string $toEncoding, string $fromEncoding): string
    {
        $converted = @iconv($fromEncoding, $toEncoding, $string);
        if ($converted === false) {
            return $string;
        }

        return $converted;
    }
}

if (!function_exists('mb_ord')) {
    function mb_ord(string $char, string $encoding = 'UTF-8'): int
    {
        if ($char === '') {
            return 0;
        }

        $result = unpack('N', mb_convert_encoding($char, 'UCS-4BE', $encoding));
        return $result[1];
    }
}

if (!function_exists('mb_chr')) {
    function mb_chr(int $codepoint, string $encoding = 'UTF-8'): string
    {
        return mb_convert_encoding(pack('N', $codepoint), $encoding, 'UCS-4BE');
    }
}

if (!function_exists('mb_internal_encoding')) {
    function mb_internal_encoding(string $encoding): bool
    {
        return true;
    }
}

if (function_exists('mb_internal_encoding')) {
    mb_internal_encoding('UTF-8');
}
header('Content-Type: application/json; charset=utf-8');

$text = resolveInputText();

if ($text === null || trim($text) === '') {
    respond(false, [], 'Missing or empty \'text\' parameter.');
}

$result = generateVariants($text);
respond(true, $result);

function resolveInputText(): ?string
{
    $method = strtoupper($_SERVER['REQUEST_METHOD'] ?? 'GET');

    if ($method === 'GET') {
        return isset($_GET['text']) ? trim((string)$_GET['text']) : null;
    }

    if ($method === 'POST') {
        if (isset($_POST['text'])) {
            return trim((string)$_POST['text']);
        }

        $raw = file_get_contents('php://input');
        if ($raw !== false && $raw !== '') {
            $decoded = json_decode($raw, true);
            if (json_last_error() === JSON_ERROR_NONE && isset($decoded['text'])) {
                return trim((string)$decoded['text']);
            }
        }

        if (isset($_GET['text'])) {
            return trim((string)$_GET['text']);
        }
    }

    return null;
}

function respond(bool $ok, array $result, string $error = ''): void
{
    $payload = [
        'ok' => $ok,
        'dev' => '@OnyxApiTeam',
    ];
    if ($ok) {
        $payload['result'] = $result;
    }
    if ($error !== '') {
        $payload['error'] = $error;
    }

    echo json_encode(
        $payload,
        JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT
    );
    exit;
}

function generateVariants(string $text): array
{
    $text = trim($text);
    $maps = loadFontMaps();
    $variants = [];
    $append = static function (?string $value) use (&$variants): void {
        if ($value === null || $value === '') {
            return;
        }

        $variants[(string)(count($variants) + 1)] = $value;
    };

    $append(mapText($text, $maps['D']));
    $append(mapText($text, $maps['L']));
    $append(bubbleRibbon($text));
    $append(mapText($text, $maps['K']));
    $append(applyScriptCalligraphy($text));
    $append(mapText($text, $maps['u']));
    $append(mapText($text, $maps['H']));
    $append(mapText($text, $maps['X']));
    $append(kawaiiBanner($text));
    $append(applyTinyCaps($text));
    $append(flipText($text));
    $append(roundSquareText($text));
    $append(squareText($text));
    $append(mapText($text, $maps['C']));
    $append(mirrorText($text));
    $append(creepifyText($text));
    $append(mapText($text, $maps['F']));
    $append(mapText($text, $maps['q']));
    $append(mapText($text, $maps['k']));
    $append(bubbleText($text));
    $append(mapText($text, $maps['z']));
    $append(mapText($text, $maps['A']));
    $append(mapText($text, $maps['v']));
    $append(mapText($text, $maps['g']));
    $append(mapText($text, $maps['f']));
    $append(mapText($text, $maps['p']));
    $append(mapText($text, $maps['b']));
    $append(mapText($text, $maps['M']));
    $append(mapText($text, $maps['j']));
    $append(mapText($text, $maps['w']));
    $append(mapText($text, $maps['y']));
    $append(mapText($text, $maps['S']));
    $append(mapText($text, $maps['I']));
    $append(mapText($text, $maps['Z']));
    $append(mapText($text, $maps['B']));
    $append(mapText($text, $maps['T']));
    $append(mapText($text, $maps['l']));
    $append(mapText($text, $maps['m']));
    $append(wrapNonWhitespace($text, '【', '】'));
    $append(wrapNonWhitespace($text, '『', '』'));
    $append(repeatBetween($text, '≋'));
    $append(repeatBetween($text, '░'));
    $append('(っ◔◡◔)っ ♥ ' . $text . ' ♥');
    $append(vaporwaveBlock($text));
    for ($i = 0; $i < 3; $i++) {
        $append(templateWrap($text));
    }
    $append(wrapDecorated($text, '[̲̅', ']'));
    $append(glitchTrail($text, '҉', true));
    $append(mapText($text, $maps['G']));
    $append(mapText($text, $maps['x']));
    $append(mapText($text, $maps['d']));
    $append(strikeThrough($text, '̶'));
    $append(strikeThrough($text, '̴'));
    $append(strikeThrough($text, '̷'));
    $append(strikeThrough($text, '̲'));
    $append(strikeThrough($text, '̳'));
    $append(strikeThrough($text, '̾'));
    $append(heartsJoin($text, '♥', true));
    $append(strikeThrough($text, '͎'));
    $append(strikeThrough($text, '͓̽'));
    $append(kawaiiBanner($text));
    $append(bubbleRibbon($text));

    return $variants;
}

function loadFontMaps(): array
{
    static $maps = null;

    if ($maps === null) {
        $maps = require __DIR__ . '/data/fontMaps.php';
    }

    return $maps;
}

function mapText(string $text, array $map): string
{
    $result = '';
    foreach (splitGraphemes($text) as $grapheme) {
        $lower = mb_strtolower($grapheme, 'UTF-8');
        $result .= $map[$grapheme] ?? $map[$lower] ?? $grapheme;
    }

    return $result;
}

function splitGraphemes(string $text): array
{
    if (function_exists('grapheme_split')) {
        $chunks = grapheme_split($text);
        if ($chunks !== false) {
            return $chunks;
        }
    }

    $parts = preg_split('//u', $text, -1, PREG_SPLIT_NO_EMPTY);
    return $parts === false ? [] : $parts;
}

function reverseGraphemes(string $text): string
{
    $chunks = splitGraphemes($text);
    return implode('', array_reverse($chunks));
}

function applyScriptCalligraphy(string $text): string
{
    $map = scriptMap();
    $result = '';
    foreach (splitGraphemes($text) as $char) {
        $lower = mb_strtolower($char, 'UTF-8');
        $result .= $map[$char] ?? $map[$lower] ?? $char;
    }

    return $result;
}

function scriptMap(): array
{
    static $map = null;

    if ($map !== null) {
        return $map;
    }

    $map = [
        '0' => '𝟢',
        '1' => '𝟣',
        '2' => '𝟤',
        '3' => '𝟥',
        '4' => '𝟦',
        '5' => '𝟧',
        '6' => '𝟨',
        '7' => '𝟩',
        '8' => '𝟪',
        '9' => '𝟫',
        'a' => '𝒶',
        'b' => '𝒷',
        'c' => '𝒸',
        'd' => '𝒹',
        'e' => '𝑒',
        'f' => '𝒻',
        'g' => '𝑔',
        'h' => '𝒽',
        'i' => '𝒾',
        'j' => '𝒿',
        'k' => '𝓀',
        'l' => '𝓁',
        'm' => '𝓂',
        'n' => '𝓃',
        'o' => '𝑜',
        'p' => '𝓅',
        'q' => '𝓆',
        'r' => '𝓇',
        's' => '𝓈',
        't' => '𝓉',
        'u' => '𝓊',
        'v' => '𝓋',
        'w' => '𝓌',
        'x' => '𝓍',
        'y' => '𝓎',
        'z' => '𝓏',
        'A' => '𝒜',
        'B' => '𝐵',
        'C' => '𝒞',
        'D' => '𝒟',
        'E' => '𝐸',
        'F' => '𝐹',
        'G' => '𝒢',
        'H' => '𝐻',
        'I' => '𝐼',
        'J' => '𝒥',
        'K' => '𝒦',
        'L' => '𝐿',
        'M' => '𝑀',
        'N' => '𝒩',
        'O' => '𝒪',
        'P' => '𝒫',
        'Q' => '𝒬',
        'R' => '𝑅',
        'S' => '𝒮',
        'T' => '𝒯',
        'U' => '𝒰',
        'V' => '𝒱',
        'W' => '𝒲',
        'X' => '𝒳',
        'Y' => '𝒴',
        'Z' => '𝒵',
    ];

    return $map;
}

function applyTinyCaps(string $text): string
{
    $map = tinyMap();
    $result = '';
    $upper = mb_strtoupper($text, 'UTF-8');
    foreach (splitGraphemes($upper) as $char) {
        $result .= $map[$char] ?? $char;
    }

    return $result;
}

function tinyMap(): array
{
    static $map = null;

    if ($map !== null) {
        return $map;
    }

    $map = [
        'A' => 'ᴀ',
        'B' => 'ʙ',
        'C' => 'ᴄ',
        'D' => 'ᴅ',
        'E' => 'ᴇ',
        'F' => 'ꜰ',
        'G' => 'ɢ',
        'H' => 'ʜ',
        'I' => 'ɪ',
        'J' => 'ᴊ',
        'K' => 'ᴋ',
        'L' => 'ʟ',
        'M' => 'ᴍ',
        'N' => 'ɴ',
        'O' => 'ᴏ',
        'P' => 'ᴘ',
        'Q' => 'Q',
        'R' => 'ʀ',
        'S' => 'ꜱ',
        'T' => 'ᴛ',
        'U' => 'ᴜ',
        'V' => 'ᴠ',
        'W' => 'ᴡ',
        'X' => 'x',
        'Y' => 'ʏ',
        'Z' => 'ᴢ',
    ];

    return $map;
}

function flipText(string $text): string
{
    $map = flipMap();
    $chars = splitGraphemes($text);
    $buffer = [];
    $combining = ['̤', '̗', '̖', '̮'];

    foreach ($chars as $index => $char) {
        if ($index > 0 && in_array($char, $combining, true)) {
            $previous = $chars[$index - 1] ?? '';
            $compound = $previous . $char;
            if (isset($map[$compound]) && !empty($buffer)) {
                $buffer[\count($buffer) - 1] = $map[$compound];
                continue;
            }
        }

        $buffer[] = $map[$char] ?? $char;
    }

    return implode('', array_reverse($buffer));
}

function flipMap(): array
{
    static $map = null;

    if ($map !== null) {
        return $map;
    }

    $map = [
        'a' => 'ɐ',
        'b' => 'q',
        'c' => 'ɔ',
        'd' => 'p',
        'e' => 'ǝ',
        'f' => 'ɟ',
        'g' => 'ɓ',
        'h' => 'ɥ',
        'i' => 'ı',
        'j' => 'ɾ',
        'k' => 'ʞ',
        'l' => 'l',
        'm' => 'ɯ',
        'n' => 'u',
        'r' => 'ɹ',
        't' => 'ʇ',
        'v' => 'ʌ',
        'w' => 'ʍ',
        'y' => 'ʎ',
        'A' => '∀',
        'B' => 'ᙠ',
        'C' => 'Ɔ',
        'D' => 'ᗡ',
        'E' => 'Ǝ',
        'F' => 'Ⅎ',
        'G' => '⅁',
        'J' => 'ſ',
        'K' => '⋊',
        'L' => '˥',
        'M' => 'W',
        'P' => 'Ԁ',
        'Q' => 'Ό',
        'R' => 'ᴚ',
        'T' => '⊥',
        'U' => '∩',
        'V' => 'Λ',
        'Y' => '⅄',
        '1' => '⇂',
        '2' => 'ᄅ',
        '3' => 'Ɛ',
        '4' => 'ㄣ',
        '5' => 'ގ',
        '6' => '9',
        '7' => 'ㄥ',
        '&' => '⅋',
        '.' => '˙',
        '"' => '„',
        ';' => '؛',
        '[' => ']',
        '(' => ')',
        '{' => '}',
        '?' => '¿',
        '!' => '¡',
        '\'' => ',',
        '<' => '>',
        '‾' => '_',
        '¯' => '_',
        '‿' => '⁀',
        '⁅' => '⁆',
        '∴' => '∵',
        "\r" => "\n",
        'ß' => 'ᙠ',
        '̈' => '̤',
        'ä' => 'ɐ̤',
        'ö' => 'o̤',
        'ü' => 'n̤',
        'Ä' => '∀̤',
        'Ö' => 'O̤',
        'Ü' => '∩̤',
        '´' => ' ̗',
        'é' => 'ǝ̗',
        'á' => 'ɐ̗',
        'ó' => 'o̗',
        'ú' => 'n̗',
        'É' => 'Ǝ̗',
        'Á' => '∀̗',
        'Ó' => 'O̗',
        'Ú' => '∩̗',
        '`' => ' ̖',
        'è' => 'ǝ̖',
        'à' => 'ɐ̖',
        'ò' => 'o̖',
        'ù' => 'n̖',
        'È' => 'Ǝ̖',
        'À' => '∀̖',
        'Ò' => 'O̖',
        'Ù' => '∩̖',
        '^' => ' ̮',
        'ê' => 'ǝ̮',
        'â' => 'ɐ̮',
        'ô' => 'o̮',
        'û' => 'n̮',
        'Ê' => 'Ǝ̮',
        'Â' => '∀̮',
        'Ô' => 'O̮',
        'Û' => '∩̮',
    ];

    foreach ($map as $key => $value) {
        if (!isset($map[$value])) {
            $map[$value] = $key;
        }
    }

    return $map;
}

function mirrorText(string $text): string
{
    $map = mirrorMap();
    $combining = ['̈', '̀', '́', '̂'];
    $normalized = str_replace("\r", "\n", $text);
    $lines = explode("\n", $normalized);
    $output = [];

    foreach ($lines as $line) {
        $chars = splitGraphemes($line);
        $buffer = [];
        foreach ($chars as $index => $char) {
            if ($index > 0 && in_array($char, $combining, true)) {
                $previous = $chars[$index - 1] ?? '';
                $compound = $previous . $char;
                if (isset($map[$compound]) && !empty($buffer)) {
                    $buffer[count($buffer) - 1] = $map[$compound];
                    continue;
                }
            }

            $lower = mb_strtolower($char, 'UTF-8');
            $buffer[] = $map[$char] ?? $map[$lower] ?? $char;
        }

        $output[] = implode('', array_reverse($buffer));
    }

    return implode("\n", $output);
}

function mirrorMap(): array
{
    static $map = null;

    if ($map !== null) {
        return $map;
    }

    $map = [
        'a' => 'ɒ',
        'b' => 'd',
        'c' => 'ɔ',
        'e' => 'ɘ',
        'f' => 'Ꮈ',
        'g' => 'ǫ',
        'h' => 'ʜ',
        'j' => 'ꞁ',
        'k' => 'ʞ',
        'l' => '|',
        'n' => 'ᴎ',
        'p' => 'q',
        'r' => 'ɿ',
        's' => 'ꙅ',
        't' => 'ƚ',
        'y' => 'ʏ',
        'z' => 'ƹ',
        'B' => 'ᙠ',
        'C' => 'Ɔ',
        'D' => 'ᗡ',
        'E' => 'Ǝ',
        'F' => 'ꟻ',
        'G' => 'Ꭾ',
        'J' => 'Ⴑ',
        'K' => '⋊',
        'L' => '⅃',
        'N' => 'Ͷ',
        'P' => 'ꟼ',
        'Q' => 'Ọ',
        'R' => 'Я',
        'S' => 'Ꙅ',
        'Z' => 'Ƹ',
        '[' => ']',
        '(' => ')',
        '{' => '}',
        '?' => '⸮',
        '<' => '>',
        'ä' => 'ɒ̈',
        'ß' => 'ᙠ',
        '´' => '`',
        'é' => 'ɘ̀',
        'á' => 'ɒ̀',
        'ó' => 'ò',
        'ú' => 'ù',
        'É' => 'Ǝ̀',
        'Á' => 'À',
        'Ó' => 'Ò',
        'Ú' => 'Ù',
        '`' => '´',
        'è' => 'ɘ́',
        'à' => 'ɒ́',
        'È' => 'Ǝ́',
        'ê' => 'ɘ̂',
        'â' => 'ɒ̂',
        'Ê' => 'Ǝ̂',
        'Ø' => 'ᴓ',
        'ø' => 'ᴓ',
    ];

    foreach ($map as $key => $value) {
        if (!isset($map[$value])) {
            $map[$value] = $key;
        }
    }

    return $map;
}

function bubbleText(string $text): string
{
    $map = bubbleMap();
    $result = '';

    foreach (splitGraphemes($text) as $char) {
        if (isset($map[$char])) {
            $result .= $map[$char];
            continue;
        }

        $code = mb_ord($char);
        if ($code >= 33) {
            $result .= $char . mb_chr(8413);
        } else {
            $result .= $char;
        }
    }

    return $result;
}

function bubbleMap(): array
{
    static $map = null;

    if ($map !== null) {
        return $map;
    }

    $map = ['0' => '⓪'];

    for ($code = 49; $code <= 57; $code++) {
        $map[chr($code)] = mb_chr($code + 9263);
    }

    for ($code = 65; $code <= 90; $code++) {
        $map[chr($code)] = mb_chr($code + 9333);
    }

    for ($code = 97; $code <= 122; $code++) {
        $map[chr($code)] = mb_chr($code + 9327);
    }

    return $map;
}

function squareText(string $text): string
{
    return appendMark($text, mb_chr(8414));
}

function roundSquareText(string $text): string
{
    return appendMark($text, mb_chr(8419));
}

function appendMark(string $text, string $mark): string
{
    $result = '';
    foreach (splitGraphemes($text) as $char) {
        $code = mb_ord($char);
        $result .= $code >= 33 ? $char . $mark : $char;
    }

    return $result;
}

function creepifyText(string $text): string
{
    $sets = creepifySets();
    $options = ['top' => true, 'middle' => true, 'bottom' => true, 'maxHeight' => 10, 'randomization' => 100];
    $result = '';

    foreach (splitGraphemes($text) as $char) {
        $glyph = $char;

        if ($options['middle'] && !empty($sets['middle'])) {
            $glyph .= randomPick($sets['middle']);
        }

        if ($options['top'] && !empty($sets['top'])) {
            $count = creepifyCount($options);
            for ($i = 0; $i < $count; $i++) {
                $glyph .= randomPick($sets['top']);
            }
        }

        if ($options['bottom'] && !empty($sets['bottom'])) {
            $count = creepifyCount($options);
            for ($i = 0; $i < $count; $i++) {
                $glyph .= randomPick($sets['bottom']);
            }
        }

        $result .= $glyph;
    }

    return $result;
}

function creepifyCount(array $options): int
{
    $spread = max(1, (int)round(($options['randomization'] / 100) * $options['maxHeight']));
    $value = max(0, $options['maxHeight'] - random_int(0, $spread));
    return max(1, $value);
}

function creepifySets(): array
{
    static $sets = null;

    if ($sets !== null) {
        return $sets;
    }

    $top = [];
    $middle = [];
    $bottom = [];

    for ($code = 768; $code <= 789; $code++) {
        $top[] = mb_chr($code);
    }

    for ($code = 790; $code <= 819; $code++) {
        if ($code === 794 || $code === 795) {
            continue;
        }

        $bottom[] = mb_chr($code);
    }

    $top[] = mb_chr(794);
    $top[] = mb_chr(795);

    for ($code = 820; $code <= 824; $code++) {
        $middle[] = mb_chr($code);
    }

    for ($code = 825; $code <= 828; $code++) {
        $bottom[] = mb_chr($code);
    }

    for ($code = 829; $code <= 836; $code++) {
        $top[] = mb_chr($code);
    }

    $top[] = mb_chr(836);
    $bottom[] = mb_chr(837);
    $top[] = mb_chr(838);
    $bottom[] = mb_chr(839);
    $bottom[] = mb_chr(840);
    $bottom[] = mb_chr(841);
    $top[] = mb_chr(842);
    $top[] = mb_chr(843);
    $top[] = mb_chr(844);
    $bottom[] = mb_chr(845);
    $bottom[] = mb_chr(846);
    $top[] = mb_chr(848);
    $top[] = mb_chr(849);
    $top[] = mb_chr(850);
    $bottom[] = mb_chr(851);
    $bottom[] = mb_chr(852);
    $bottom[] = mb_chr(853);
    $bottom[] = mb_chr(854);
    $top[] = mb_chr(855);
    $top[] = mb_chr(856);
    $bottom[] = mb_chr(857);
    $bottom[] = mb_chr(858);
    $top[] = mb_chr(859);
    $bottom[] = mb_chr(860);
    $top[] = mb_chr(861);
    $top[] = mb_chr(861);
    $bottom[] = mb_chr(863);
    $top[] = mb_chr(864);
    $top[] = mb_chr(865);

    $sets = ['top' => $top, 'middle' => $middle, 'bottom' => $bottom];
    return $sets;
}

function wrapNonWhitespace(string $text, string $prefix, string $suffix): string
{
    return preg_replace('/([^\s])/u', $prefix . '$1' . $suffix, $text) ?? $text;
}

function repeatBetween(string $text, string $glue): string
{
    $chars = splitGraphemes($text);
    if ($chars === []) {
        return '';
    }

    return $glue . implode($glue, $chars) . $glue;
}

function wrapDecorated(string $text, string $prefix, string $suffix): string
{
    $result = '';
    foreach (splitGraphemes($text) as $char) {
        if (preg_match('/\s/u', $char)) {
            $result .= $char;
            continue;
        }

        $result .= $prefix . $char . $suffix;
    }

    return $result;
}

function glitchTrail(string $text, string $mark, bool $appendTrail = false): string
{
    $chars = splitGraphemes($text);
    if ($chars === []) {
        return '';
    }

    $result = implode($mark, $chars);
    return $appendTrail ? $result . $mark : $result;
}

function strikeThrough(string $text, string $mark): string
{
    $result = '';
    foreach (splitGraphemes($text) as $char) {
        $result .= $char . $mark;
    }

    return $result;
}

function heartsJoin(string $text, string $mark, bool $append = false): string
{
    $chars = splitGraphemes($text);
    if ($chars === []) {
        return '';
    }

    $result = implode($mark, $chars);
    return $append ? $result . $mark : $result;
}

function bubbleRibbon(string $text): string
{
    if (trim($text) === '') {
        return '';
    }

    $padding = randomEmojiTrail(2);
    return $padding . '  ' . accentDecorated($text) . '  ' . $padding;
}

function accentDecorated(string $text): string
{
    $result = '';
    foreach (splitGraphemes($text) as $char) {
        $result .= accentChar($char);
    }

    return $result;
}

function accentChar(string $char): string
{
    $map = accentMap();
    $lower = mb_strtolower($char, 'UTF-8');
    $value = $map[$char] ?? $map[$lower] ?? null;

    if ($value === null) {
        return $char;
    }

    if (is_array($value)) {
        return randomPick($value);
    }

    return $value;
}

function accentMap(): array
{
    static $map = null;

    if ($map !== null) {
        return $map;
    }

    $map = [
        '&' => '⅋',
        '%' => ['⅍', '℀', '℁', '℆', '℅'],
        '0' => ['０', 'Ѳ', 'ʘ'],
        '1' => ['➀', '❶', '１'],
        '2' => ['２', '❷', '➁'],
        '3' => ['３', '❸', '➂'],
        '4' => ['４', '❹', '➃'],
        '5' => ['❺', '➄', '５'],
        '6' => ['６', '❻', '➅'],
        '7' => ['７', '❼', '➆'],
        '8' => ['８', '➇', '❽'],
        '9' => ['➈', '❾', '９'],
        '<' => ['≼', '≺', '≪', '☾', '≾', '⋜', '⋞', '⋐', '⊂', '⊏', '⊑', '《', '＜', '❮', '❰', '⫷'],
        '>' => ['☽', '≫', '≻', '≽', '≿', '⋝', '⋟', '⋑', '⊃', '⊐', '⊒', '⫸', '》', '＞', '❯', '❱'],
        '[' => ['【', '〖', '〘', '〚', '［'],
        ']' => ['】', '〗', '〙', '〛', '］'],
        '*' => ['✨', '✩', '✪', '✫', '✬', '✭', '✮', '✯', '✰', '✦', '✱', '✲', '✳', '✴', '✵', '✶', '✷', '֍', '֎', '✸', '✹', '✺', '✻', '✼', '✽', '✾', '✿', '❀', '❁', '❂', '❃', '❄', '★', '☆', '＊'],
        'a' => ['Ⓐ', 'ⓐ', 'α', 'Ａ', 'ａ', 'ᗩ', '卂', 'Δ', 'ค', 'ά', 'Ã', '𝔞', '𝓪', '𝒶', '𝓐', '𝐀', '𝐚', '𝔸', '𝕒', 'ᵃ'],
        'b' => ['Ⓑ', 'ⓑ', 'в', 'Ｂ', '乃', 'ｂ', 'ᗷ', 'β', '๒', '𝔟', '𝓫', '𝒷', '𝓑', '𝐁', '𝐛', '𝔹', '𝕓', 'ᵇ'],
        'c' => ['Ⓒ', 'ⓒ', '匚', '¢', 'Ｃ', 'ｃ', 'ᑕ', 'Ć', 'ς', 'Č', '℃', '𝔠', '𝓬', '𝒸', '𝓒', '𝐂', '𝐜', 'ℂ', '𝕔', 'ᶜ'],
        'd' => ['Ⓓ', 'ⓓ', '∂', 'Ｄ', 'ｄ', 'ᗪ', 'Đ', '๔', 'Ď', '𝔡', '𝓭', '𝒹', '𝓓', '𝐃', '𝐝', '𝔻', '𝕕', 'ᵈ'],
        'e' => ['Ⓔ', '乇', 'ⓔ', 'є', 'Ｅ', 'ｅ', 'ᗴ', '€', 'ε', 'έ', 'Ẹ', '𝔢', '𝒆', '𝑒', '𝓔', '𝐄', '𝐞', '𝔼', '𝕖', 'ᵉ'],
        'f' => ['Ⓕ', 'ⓕ', 'ƒ', 'Ｆ', 'ｆ', '千', 'ᖴ', 'ℱ', 'Ŧ', 'Ƒ', '𝔣', '𝒇', '𝒻', '𝓕', '𝐅', '𝐟', '𝔽', '𝕗', 'ᶠ'],
        'g' => ['Ⓖ', 'ⓖ', 'ق', 'Ｇ', 'ｇ', 'Ǥ', 'Ꮆ', 'ﻮ', 'ģ', 'Ğ', '𝔤', '𝓰', '𝑔', '𝓖', '𝐆', '𝐠', '𝔾', '𝕘', 'ᵍ', 'Ꮆ'],
        'h' => ['Ⓗ', '卄', 'ⓗ', 'н', 'Ｈ', 'ｈ', 'ᕼ', 'Ħ', 'ђ', 'ħ', 'Ĥ', '𝔥', '𝓱', '𝒽', '𝓗', '𝐇', '𝐡', 'ℍ', '𝕙', 'ʰ'],
        'i' => ['Ⓘ', 'ⓘ', 'ι', 'Ｉ', 'ｉ', 'Ꭵ', '丨', 'Ɨ', 'ί', 'Į', '𝔦', '𝓲', '𝒾', '𝓘', '𝐈', '𝐢', '𝕀', '𝕚', 'ᶤ'],
        'j' => ['Ⓙ', 'ⓙ', 'נ', 'Ｊ', 'ｊ', 'ᒎ', 'Ĵ', 'ן', 'Ĵ', '𝔧', '𝓳', '𝒿', '𝓙', '𝐉', '𝐣', '𝕁', '𝕛', 'ʲ'],
        'k' => ['Ⓚ', 'ⓚ', 'к', 'Ｋ', 'ｋ', 'ᛕ', 'Ҝ', 'ķ', 'Ќ', '𝔨', '𝓴', '𝓀', '𝓚', '𝐊', '𝐤', '𝕂', '𝕜', 'ᵏ'],
        'l' => ['Ⓛ', 'ⓛ', 'ℓ', 'ㄥ', 'Ｌ', 'ｌ', 'ᒪ', 'Ł', 'ℓ', 'Ļ', 'Ĺ', '𝔩', '𝓵', '𝓁', '𝓛', '𝐋', '𝐥', '𝕃', '𝕝', 'ˡ'],
        'm' => ['Ⓜ', 'ⓜ', 'м', 'Ｍ', 'ｍ', 'ᗰ', 'Μ', '๓', 'ϻ', '𝔪', '𝓶', '𝓂', '𝓜', '𝐌', '𝐦', '𝕄', '𝕞', 'ᵐ', '爪'],
        'n' => ['Ⓝ', '几', 'ⓝ', 'η', 'Ｎ', 'ｎ', 'ᑎ', 'Ň', 'η', 'Ň', '𝔫', '𝓷', '𝓃', '𝓝', '𝐍', '𝐧', 'ℕ', '𝕟', 'ᶰ'],
        'o' => ['Ⓞ', 'ㄖ', 'ⓞ', 'σ', 'Ｏ', 'ｏ', 'ᗝ', 'Ø', '๏', 'ό', 'Ỗ', '𝔬', '𝓸', '𝑜', '𝓞', '𝐎', '𝐨', '𝕆', '𝕠', 'ᵒ'],
        'p' => ['Ⓟ', 'ⓟ', 'ρ', 'Ｐ', 'ｐ', '卩', 'ᑭ', 'Ƥ', 'ק', 'ρ', '𝔭', '𝓹', '𝓅', '𝓟', '𝐏', '𝐩', 'ℙ', '𝕡', 'ᵖ'],
        'q' => ['Ⓠ', 'ⓠ', 'q', 'Ｑ', 'ｑ', 'Ɋ', 'Ω', 'ợ', 'Ǫ', '𝔮', '𝓺', '𝓆', '𝓠', '𝐐', '𝐪', 'ℚ', '𝕢', 'ᵠ'],
        'r' => ['Ⓡ', 'ⓡ', 'я', '尺', 'Ｒ', 'ｒ', 'ᖇ', 'Ř', 'я', 'ŕ', '𝔯', '𝓻', '𝓇', '𝓡', '𝐑', '𝐫', 'ℝ', '𝕣', 'ʳ'],
        's' => ['Ⓢ', 'ⓢ', 'ѕ', 'Ｓ', '丂', 'ｓ', 'ᔕ', 'Ş', 'ร', 'ş', 'Ŝ', '𝔰', '𝓼', '𝓈', '𝓢', '𝐒', '𝐬', '𝕊', '𝕤', 'ˢ'],
        't' => ['Ⓣ', 'ⓣ', 'т', 'Ｔ', 'ｔ', '丅', 'Ŧ', 'ţ', 'Ť', '𝔱', '𝓽', '𝓉', '𝓣', '𝐓', '𝐭', '𝕋', '𝕥', 'ᵗ'],
        'u' => ['Ⓤ', 'ⓤ', 'υ', 'Ｕ', 'ｕ', 'ᑌ', 'Ữ', 'ย', 'ù', 'Ǘ', '𝔲', '𝓾', '𝓊', '𝓤', '𝐔', '𝐮', '𝕌', '𝕦', 'ᵘ'],
        'v' => ['Ⓥ', 'ⓥ', 'ν', 'Ｖ', 'ｖ', 'ᐯ', 'V', 'ש', 'ν', 'Ѷ', '𝔳', '𝓿', '𝓋', '𝓥', '𝐕', '𝐯', '𝕍', '𝕧', 'ᵛ'],
        'w' => ['Ⓦ', 'ⓦ', 'ω', 'Ｗ', 'ｗ', 'ᗯ', 'Ŵ', 'ฬ', 'ώ', '𝔴', '𝔀', '𝓌', '𝓦', '𝐖', '𝐰', '𝕎', '𝕨', 'ʷ', '山'],
        'x' => ['Ⓧ', 'ⓧ', 'χ', 'Ｘ', '乂', 'ｘ', '᙭', 'Ж', 'א', '𝔵', '𝔁', '𝓍', '𝓧', '𝐗', '𝐱', '𝕏', '𝕩', 'ˣ'],
        'y' => ['Ⓨ', 'ㄚ', 'ⓨ', 'у', 'Ｙ', 'ｙ', 'Ƴ', '¥', 'ץ', 'ү', 'ч', 'Ў', '𝔶', '𝔂', '𝓎', '𝓨', '𝐘', '𝐲', '𝕐', '𝕪', 'ʸ'],
        'z' => ['Ⓩ', 'ⓩ', 'z', '乙', 'Ｚ', 'ｚ', 'Ƶ', 'Ž', 'ż', '𝔷', '𝔃', '𝓏', '𝓩', '𝐙', '𝐳', 'ℤ', '𝕫', 'ᶻ'],
    ];

    return $map;
}

function randomEmojiTrail(int $count): string
{
    $pool = emojiPool();
    $result = [];
    for ($i = 0; $i < $count; $i++) {
        $result[] = randomPick($pool);
    }

    return implode('', $result);
}

function emojiPool(): array
{
    return ['🐙', '🐉', '🐊', '🐒', '🐝', '🐜', '🐚', '🐲', '🐳', '🐸', '👑', '👹', '👺', '👤', '💲', '💣', '💙', '💚', '💛', '💜', '💝', '💗', '💘', '💞', '💔', '💥', '🐯', '🐼', '🐻', '🐺', '👌', '🐍', '🐧', '🐟', '🐠', '🐨', '🎯', '🏆', '🎁', '🎀', '🎉', '🎈', '🍮', '🍭', '🍬', '🍫', '🍪', '🍧', '🌷', '🍓', '😺', '😾', '✎', '😎', '😝', '😂', '😈', '😡', '😲', '😳', '🍔', '🍟', '🍩', '🎃', '🎄', '🎅', '🐣', '🐤', '👍', '👊', '👻', '👽', '👮', '💎', '💋', '👣', '💀', '💢', '🔥', '♔', '♕', '♖', '♗', '♘', '♙', '♚', '♛', '♜', '♝', '♞', '♟', '♠', '♡', '♢', '♣', '♤', '♥', '♦', '♧', '♨', '♩', '♪', '♬', '★', '☆', '☺', '☹', '☯', '☮', '☢', '☠', '☟', '☞', '☝', '☜', '✌', '✋', '✊', '⛵', 'ൠ', '✌', 'ඏ'];
}

function randomPick(array $items)
{
    if ($items === []) {
        return '';
    }

    $index = random_int(0, count($items) - 1);
    return $items[$index];
}

function vaporwaveBlock(string $text): string
{
    $content = trim($text);
    if ($content === '') {
        return '';
    }

    $fullwidth = mapText($content, loadFontMaps()['e']);
    $wordCount = max(1, count(preg_split('/\s+/u', trim($content)) ?: []));
    $kana = vaporwaveString(max(3, $wordCount));
    $line1 = $fullwidth . '　' . $kana;
    $line2 = vaporwaveGlitch($fullwidth) . '　（' . $kana . '）';
    $line3 = '【' . $fullwidth . '】';

    return implode("\n\n", [$line1, $line2, $line3]);
}

function vaporwaveGlitch(string $text): string
{
    $result = str_replace('　', '░', $text);
    $result = str_replace('ａｅ', 'æ', $result);
    $result = str_replace('Ａ', 'Λ', $result);
    $result = preg_replace_callback('/Ｅ/iu', static function (): string {
        return random_int(0, 1) ? 'Ξ' : 'Σ';
    }, $result) ?? $result;
    $result = str_replace('Ｏ', '♢', $result);

    return $result;
}

function vaporwaveString(int $length): string
{
    $pool = vaporwavePool();
    $characters = preg_split('//u', $pool, -1, PREG_SPLIT_NO_EMPTY) ?: [];
    $maxIndex = count($characters) - 1;
    $result = '';

    for ($i = 0; $i < $length; $i++) {
        $result .= $characters[random_int(0, $maxIndex)];
    }

    return $result;
}

function vaporwavePool(): string
{
    return 'リサフランク現代のコンピュ竹内 まりや若者が履く流行のスニーカー真夜中のドアホットドッグマスターストライカーソニーブギ新しい日の誕生ライフ - ヒスイ蒸気波 無線゠ァアィイゥウェエォオカガキギクグケゲコゴサザシジスズセゼソゾタダチヂッツヅテデトドナニヌネノハバパヒビピフブプヘベペホボポマミムメモャヤュユョヨラリルレロヮワヰヱヲンヴヵヶヷヸヹヺ・ーヽヾヿぁあぃいぅうぇえぉおかがきぎくぐけげこごさざしじすずせぜそぞただちぢっつづてでとどなにぬねのはばぱひびぴふぶぷへべぺほぼぽまみむめもゃやゅゆょよらりるれろゎわゐゑをんゔゕゖ゙゚゛゜ゝゞゟ亜哀挨愛曖悪握圧扱宛嵐安案暗以衣位囲医依委威為畏胃尉異移萎偉椅彙意違維慰遺緯域育壱逸茨芋引印因咽姻員院淫陰飲隠韻右宇羽雨唄鬱畝浦運雲永泳英映栄営詠影鋭衛易疫益液駅悦越謁閲円延沿炎怨宴媛援園煙猿遠鉛塩演縁艶汚王凹央応往押旺欧殴桜翁奥横岡屋億憶臆虞乙俺卸音恩温穏下化火加可仮何花佳価果河苛科';
}

function templateWrap(string $text): string
{
    $frame = randomPick(templateFrames());
    return str_replace('[[text]]', $text, $frame);
}

function templateFrames(): array
{
    return [
        'ཧᜰ꙰ꦿ➢[[text]]༒',
        '꧁༒☬[[text]]☬༒꧂',
        '★彡[[text]]彡★',
        '❄️[[text]]❄️',
        '💤📿[[text]]📿💤',
        '⚡[[text]]⚡',
        '๖ۣۜ[[text]]༻࿌𖣘',
        '✞☬[[text]]☬✞',
        '✞༒[[text]]༒✞',
        '✞☠︎[[text]]☠︎✞',
        '✞✞[[text]]✞✞',
        '✞ঔৣ[[text]]ঔৣ✞',
        '༺[[text]]༻',
        '༒•[[text]]•༒',
        '꧁༒•[[text]]•༒꧂',
        '꧁꧁[[text]]꧂꧂',
        '꧁[[text]]꧂',
        '☆꧁[[text]]꧂☆',
        '☆꧁༒[[text]]༒꧂☆',
        '꧁☬[[text]]☬꧂',
        '☆꧁༒☬[[text]]☬༒꧂☆',
        '★·.·´¯`·.·★ [[text]] ★·.·´¯`·.·★',
        '▁ ▂ ▄ ▅ ▆ ▇ █ [[text]] █ ▇ ▆ ▅ ▄ ▂ ▁',
        '°°°·.°·..·°¯°·._.· [[text]] ·._.·°¯°·.·° .·°°°',
        '¸,ø¤º°`°º¤ø,¸¸,ø¤º° [[text]] °º¤ø,¸¸,ø¤º°`°º¤ø,¸',
        'ıllıllı [[text]] ıllıllı',
        '•?((¯°·._.• [[text]] •._.·°¯))؟•',
        '▌│█║▌║▌║ [[text]] ║▌║▌║█│▌',
        '×º°”˜`”°º× [[text]] ×º°”˜`”°º×',
        '•••´º´•» [[text]] «•´º´•••',
        '*•.¸♡ [[text]] ♡¸.•*',
        '╰☆☆ [[text]] ☆☆╮',
        '.•°¤*(¯`★´¯)*¤° [[text]] °¤*(¯´★`¯)*¤°•.',
        '(¯´•._.• [[text]] •._.•´¯)',
        '¸„.-•~¹°”ˆ˜¨ [[text]] ¨˜ˆ”°¹~•-.„¸',
        '░▒▓█ [[text]] █▓▒░',
        '░▒▓█►─═  [[text]] ═─◄█▓▒░',
        '★彡 [[text]] 彡★',
        '•´¯`•. [[text]] .•´¯`•',
        '§.•´¨\'°÷•..× [[text]] ×,.•´¨\'°÷•..§',
        '•°¯`•• [[text]] ••´¯°•',
        '(¯`*•.¸,¤°´✿.｡.:* [[text]] *.:｡.✿`°¤,¸.•*´¯)',
        "|!¤*'~``~'*¤!| [[text]] |!¤*'~``~'*¤!|",
        '•._.••´¯``•.¸¸.•` [[text]] `•.¸¸.•´´¯`••._.•',
        '••¤(`×¤ [[text]] ¤×´)¤••',
        '•´¯`•» [[text]] «•´¯`•',
        ' .o0×X×0o. [[text]] .o0×X×0o.',
        '¤¸¸.•´¯`•¸¸.•..>> [[text]] <<..•.¸¸•´¯`•.¸¸¤',
        '—(••÷ [[text]] ÷••)—',
        '`•.¸¸.•´´¯`••._.• [[text]] •._.••`¯´´•.¸¸.•`',
        ",-*'^'~*-.,_,.-*~ [[text]] ~*-.,_,.-*~'^'*-," ,
        '`•.,¸¸,.•´¯ [[text]] ¯`•.,¸¸,.•´',
        '↤↤↤↤↤ [[text]] ↦↦↦↦↦',
        '➶➶➶➶➶ [[text]] ➷➷➷➷➷',
        '↫↫↫↫↫ [[text]] ↬↬↬↬↬',
        '·.¸¸.·♩♪♫ [[text]] ♫♪♩·.¸¸.·',
        '【｡_｡】 [[text]] 【｡_｡】',
        '|I{•------» [[text]] «------•}I|',
        '▀▄▀▄▀▄ [[text]] ▄▀▄▀▄▀',
        '(-_-) [[text]] (-_-)',
        "-漫~*'¨¯¨'*·舞~ [[text]] ~舞*'¨¯¨'*·~漫-",
        '๑۞๑,¸¸,ø¤º°`°๑۩ [[text]] ๑۩ ,¸¸,ø¤º°`°๑۞๑',
        '••.•´¯`•.•• [[text]] ••.•´¯`•.••',
        '◦•●◉✿ [[text]] ✿◉●•◦',
        '╚»★«╝ [[text]] ╚»★«╝',
        '-·=»‡«=·- [[text]] -·=»‡«=·-',
        '∙∙·▫▫ᵒᴼᵒ▫ₒₒ▫ᵒᴼᵒ▫ₒₒ▫ᵒᴼᵒ [[text]] ᵒᴼᵒ▫ₒₒ▫ᵒᴼᵒ▫ₒₒ▫ᵒᴼᵒ▫▫·∙∙',
        '¸¸♬·¯·♩¸¸♪·¯·♫¸¸ [[text]] ¸¸♫·¯·♪¸¸♩·¯·♬¸¸',
        'ஜ۩۞۩ஜ [[text]] ஜ۩۞۩ஜ',
        '¤ (¯´☆✭.¸_)¤ [[text]] ¤(_¸.✭☆´¯) ¤',
        '(¯`·.¸¸.·´¯`·.¸¸.-> [[text]] <-.¸¸.·´¯`·.¸¸.·´¯)',
        '✿.｡.:* ☆:**:. [[text]] .:**:.☆*.:｡.✿',
        '.•♫•♬• [[text]] •♬•♫•.',
        'ღ(¯`◕‿◕´¯) ♫ ♪ ♫ [[text]] ♫ ♪ ♫ (¯`◕‿◕´¯)ღ',
        '«-(¯`v´¯)-« [[text]] »-(¯`v´¯)-»',
    ];
}

function kawaiiBanner(string $text): string
{
    $content = trim($text);
    if ($content === '') {
        return '';
    }

    $decorators = [
        static fn(int $n): string => decorateWithPool($n, candyPool(), ' ⋆ '),
        static fn(int $n): string => decorateWithPool($n, sparklePool(), ''),
        static fn(int $n): string => decorateWithPool($n, critterPool(), ' ⋆ '),
        static fn(int $n): string => randomPick(asciiBorders()),
    ];

    $decor = $decorators[random_int(0, count($decorators) - 1)];
    $pads = $decor(random_int(1, 2));
    $body = stylizeSentences($content);

    return $pads . '  🎀  ' . $body . '  🎀  ' . reverseGraphemes($pads);
}

function decorateWithPool(int $count, array $pool, string $glue): string
{
    $chunks = [];
    for ($i = 0; $i < $count; $i++) {
        $chunks[] = randomPick($pool);
    }

    return implode($glue, $chunks);
}

function stylizeSentences(string $text): string
{
    $parts = preg_split('/([!?.]+)/u', $text, -1, PREG_SPLIT_DELIM_CAPTURE);
    if ($parts === false) {
        return $text;
    }

    $result = '';
    foreach ($parts as $part) {
        if ($part === '') {
            continue;
        }

        if (preg_match('/^[!?.]+$/u', $part)) {
            $result .= $part;
            continue;
        }

        $result .= stylizeToken($part);
    }

    return $result;
}

function stylizeToken(string $token): string
{
    $trimmed = trim($token);
    if ($trimmed === '') {
        return $token;
    }

    $token = replaceEmoticons($token);
    $token = replacePunctuationMarks($token);
    $token = decorateQuotes($token);
    $token = replaceFloralLetters($token);

    return applyScriptCalligraphy($token);
}

function replaceEmoticons(string $text): string
{
    $map = emoticonMap();
    foreach ($map as $needle => $variants) {
        $text = str_replace($needle, randomPick($variants), $text);
    }

    return $text;
}

function replacePunctuationMarks(string $text): string
{
    return preg_replace_callback('/[!?]/u', static function (array $match): string {
        $char = $match[0];
        if ($char === '!') {
            return randomPick(['❣', '❢', '!']);
        }

        return randomPick(['¿', '?']);
    }, $text) ?? $text;
}

function decorateQuotes(string $text): string
{
    static $state = 0;

    return preg_replace_callback('/"/u', static function () use (&$state): string {
        $state = ($state + 1) % 2;
        $quotes = ['❝', '❞'];
        return $quotes[$state];
    }, $text) ?? $text;
}

function replaceFloralLetters(string $text): string
{
    $pool = floralPool();
    $chars = splitGraphemes($text);
    foreach ($chars as $index => $char) {
        $lower = mb_strtolower($char, 'UTF-8');
        if ($lower === 'o' || $char === '0') {
            $chars[$index] = randomPick($pool);
        }
    }

    return implode('', $chars);
}

function emoticonMap(): array
{
    return [
        ':)' => ['٩(˘◡˘)۶', '(•◡•)', '(ɔ◔‿◔)ɔ', '❀◕‿◕❀', '(｡◕‿◕｡)', '●‿●', '◕‿◕', '😍', 'ツ', '(✿ヘᴥヘ)', '🌘‿🌘', '(づ｡◕‿‿◕｡)づ', '(/◔◡◔)/', 's(^‿^)-b', '(人◕‿◕)', '(✿╹◡╹)', '◔◡◔'],
        '(:' => ['٩(˘◡˘)۶', '(•◡•)', '(ɔ◔‿◔)ɔ', '❀◕‿◕❀', '(｡◕‿◕｡)', '●‿●', '◕‿◕', '😍', 'ツ', '(✿ヘᴥヘ)', '🌘‿🌘', '(づ｡◕‿‿◕｡)づ', '(/◔◡◔)/', 's(^‿^)-b', '(人◕‿◕)', '(✿╹◡╹)', '◔◡◔'],
        '^^' => ['(^▽^)', '(✿^▽^)', 'ᵔ⌣ᵔ', 'ᵔᴥᵔ', '(≧◡≦)', '^ㅅ^', '^ㅂ^', '(⌒‿⌒)', '◠◡◠', '⁀‿⁀', ''],
        ':P' => [':Þ', '😛'],
        ':p' => [':Þ', '😛'],
        ':D' => ['(ﾉ◕ヮ◕)ﾉ*:･ﾟ✧', '(✿◕ᗜ◕)━♫.*･｡ﾟ', 'ᕕ(ᐛ)ᕗ'],
        ':d' => ['(ﾉ◕ヮ◕)ﾉ*:･ﾟ✧', '(✿◕ᗜ◕)━♫.*･｡ﾟ', 'ᕕ(ᐛ)ᕗ'],
        '<3' => ['♡', '❤', '♥', '❥', '💘', '💙', '💗', '💖', '💕', '💓', '💞', '💝', '💟'],
        '(y)' => ['👍', '👌'],
        '(Y)' => ['👍', '👌'],
    ];
}

function floralPool(): array
{
    return ['❁', '🌺', '❀', '💮', '🏵', '🌸', '☯', '😍', '♡', '🍩', '🍬', '💙', '🌞', '💍', '🍪', '❤', '💗', '🍑', '💞'];
}

function candyPool(): array
{
    return ['🍬', '🍭', '🍒', '🍎', '🍉', '🍇', '🍓', '🍌', '🍑', '🍰', '🎂', '🍩', '🍪', '🍧', '🍦', '🍫', '🍡'];
}

function sparklePool(): array
{
    return ['⚛', '🌌', '🌠', '*', ':', '｡', '･ﾟ', '✧', '✮', '★', '✩', '⋆', '.', '°', '`', '✴', '✳', '✶', '✷', '❈'];
}

function critterPool(): array
{
    return ['🐯', '🐎', '🐖', '🐷', '🐏', '🐑', '🐐', '🐫', '🐘', '🐭', '🐁', '🐀', '🐹', '🐰', '🐇', '🐿', '🐻', '🐨', '🐼', '🐾', '🐔', '🐓', '🐣', '🐤', '🐥', '🐦', '🐧', '🕊', '🐸', '🐢', '🐳', '🐋', '🐬', '🐟', '🐠', '🐡', '🐙', '🐚', '🐌', '🐞'];
}

function asciiBorders(): array
{
    return [
        '•?((¯°·._.• ',
        'ıllıllı ',
        '¸,ø¤º°`°º¤ø,¸¸,ø¤º° ',
        '°°°·.°·..·°¯°·._.· ',
        '•´¯`•. ',
        '×º°”˜`”°º× ',
        '•]••´º´•» ',
        ']|I{•------» ',
        '§.•´¨\'°÷•..× ',
        '•°¯`•• ',
        '(¯`·.¸¸.-> ',
        '*´¯`*.¸¸.*´¯`* ',
        '(¯`·.¸¸.-> °º ',
        '°·.¸.·°¯°·.¸.·°¯°·.¸.-> ',
        '•._.••´¯``•.¸¸.•` ',
        '¸„.-•~¹°”ˆ˜¨ ',
       '(¯´•._.• ',
        '••¤(`× ',
       '•´¯`•» ',
        '`•.,¸¸,.•´¯ ',
        ',-*\' ^ \'~*-.,_,.-*~ ',
        '`•.¸¸.•´´¯`••._.• ',
        '—(••÷ ',
        '¤¸¸.•´¯`•¸¸.•..>> ',
        '••.•´¯`•.•• ',
        '.•°¤*(¯`★´¯)*¤° ',
        '๑۞๑,¸¸,ø¤º°`°๑۩ ',
        '-漫~*\'¨¯¨\'*·舞~ ',
        '╚»★«╝ ',
        '-·=»‡«=·- ',
        '∙∙·▫▫ᵒᴼᵒ▫ₒₒ▫ᵒᴼᵒ▫ₒₒ▫ᵒᴼᵒ ',
        '¸¸♬·¯·♩¸¸♪·¯·♫¸¸ ',
        'ஜ۩۞۩ஜ ',
        '¤ (¯´☆✭.¸_)¤ ',
        '✿.｡.:* ☆:**:. ',
        '.•♫•♬• ',
        'ღ(¯`◕‿◕´¯) ♫ ♪ ♫ ',
        '«-(¯`v´¯)-« ',
    ];
}

