<?php

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

$giftsUrl = 'https://giftcharts-api.onrender.com/gifts';
$dollarApiUrl = 'https://onyxapi.ir/alanchand/?type=currencies&symbols=usd';

function fetchParallel($urls) {
    $multi = curl_multi_init();
    $chs = [];
    foreach ($urls as $key => $url) {
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_multi_add_handle($multi, $ch);
        $chs[$key] = $ch;
    }
    do {
        $status = curl_multi_exec($multi, $running);
        curl_multi_select($multi);
    } while ($running && $status == CURLM_OK);
    $results = [];
    foreach ($chs as $key => $ch) {
        $results[$key] = curl_multi_getcontent($ch);
        curl_multi_remove_handle($multi, $ch);
        curl_close($ch);
    }
    curl_multi_close($multi);
    return $results;
}

function getDollarPrice($json) {
    $data = json_decode($json, true);
    if (isset($data['ok'], $data['data']['usd']['sell']) && $data['ok']) {
        return $data['data']['usd']['sell'];
    }
    return null;
}

function toIrt($usd, $dollarPrice) {
    return ($usd && $dollarPrice) ? round($usd * $dollarPrice) : null;
}

$q = '';
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['q'])) {
    $q = trim($_POST['q']);
} elseif (isset($_GET['q'])) {
    $q = trim($_GET['q']);
}

list($giftsResponse, $usdResponse) = fetchParallel([$giftsUrl, $dollarApiUrl]);
$gifts = json_decode($giftsResponse, true) ?: [];
$dollarPrice = getDollarPrice($usdResponse);

if ($q !== '') {
    $filtered = array_filter($gifts, function($g) use ($q) {
        return isset($g['name']) && mb_stripos($g['name'], $q) !== false;
    });
    if (empty($filtered)) {
        echo json_encode([
            'ok'    => false,
            'dev'   => '@OnyxApiTeam',
            'error' => 'No gifts found matching the query'
        ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
        exit;
    }
    $gifts = $filtered;
}

$gifts = array_map(function ($g) use ($dollarPrice) {
    return [
        'id' => $g['_id'] ?? null,
        'name' => $g['name'] ?? null,
        'releaseDate' => $g['releaseDate'] ?? null,
        'supply' => [
            'current'  => $g['supply'] ?? null,
            'initial'  => $g['initSupply'] ?? null,
            'upgraded' => $g['upgradedSupply'] ?? null
        ],
        'prices' => [
            'current' => [
                'ton' => $g['priceTon'] ?? null,
                'usd' => $g['priceUsd'] ?? null,
                'irt' => toIrt($g['priceUsd'] ?? 0, $dollarPrice)
            ],
            'initial' => [
                'ton' => $g['initTonPrice'] ?? null,
                'usd' => $g['initUsdPrice'] ?? null
            ],
            'stars'   => $g['starsPrice'] ?? null,
            'upgrade' => $g['upgradePrice'] ?? null
        ],
        'history' => [
            '24hAgo' => [
                'ton' => $g['tonPrice24hAgo'] ?? null,
                'usd' => $g['usdPrice24hAgo'] ?? null,
                'irt' => toIrt($g['usdPrice24hAgo'] ?? 0, $dollarPrice)
            ],
            'weekAgo' => [
                'ton' => $g['tonPriceWeekAgo'] ?? null,
                'usd' => $g['usdPriceWeekAgo'] ?? null,
                'irt' => toIrt($g['usdPriceWeekAgo'] ?? 0, $dollarPrice)
            ],
            'monthAgo' => [
                'ton' => $g['tonPriceMonthAgo'] ?? null,
                'usd' => $g['usdPriceMonthAgo'] ?? null,
                'irt' => toIrt($g['usdPriceMonthAgo'] ?? 0, $dollarPrice)
            ]
        ]
    ];
}, array_values($gifts));

echo json_encode([
    'ok'    => true,
    'dev'   => '@OnyxApiTeam',
    'gifts' => $gifts
], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);

?>
