<?php

class InstagramDownloader {
    private $url;
    private $type;
    private $cookieFile;
    
    public function __construct($url) {
        $this->url = $url;
        $this->cookieFile = tempnam(sys_get_temp_dir(), 'cookie_');
    }
    
    public function __destruct() {
        if ($this->cookieFile && file_exists($this->cookieFile)) {
            @unlink($this->cookieFile);
        }
    }
    
    public function process() {
        $validation = UrlValidator::validate($this->url);
        if (!$validation["valid"]) {
            ResponseHandler::send(false, [], $validation["error"], null, 400);
        }
        
        $this->type = TypeDetector::detect($this->url);
        if (!$this->type) {
            ResponseHandler::send(false, [], "Unsupported Instagram URL format", null, 400);
        }
        
        $caption = CaptionExtractor::extract($this->url, $this->cookieFile);
        $apiUrl = TypeDetector::getApiUrl($this->type);
        
        $mediaHtml = $this->fetchMedia($apiUrl);
        if (!$mediaHtml) {
            ResponseHandler::send(false, [], "No downloadable media found", $caption, 502);
        }
        
        $html = html_entity_decode(str_replace('\"', '"', $mediaHtml));
        $result = MediaParser::parse($html, $this->type);
        
        if (empty($result)) {
            ResponseHandler::send(false, [], "No media extracted", $caption, 204);
        }
        
        ResponseHandler::send(true, $result, null, $caption);
    }
    
    private function fetchMedia($apiUrl) {
        $attempt = 0;
        
        while ($attempt < MAX_RETRY_ATTEMPTS) {
            $response = HttpClient::post($apiUrl, [
                "url" => $this->url,
                "v" => API_VERSION,
                "lang" => API_LANG
            ], [
                "Referer: https://downloadgram.org/",
                "Origin: https://downloadgram.org"
            ], $this->cookieFile);
            
            if ($response && preg_match('/innerHTML\s*=\s*"(.+?)";/s', $response, $m)) {
                return $m[1];
            }
            
            $attempt++;
            if ($attempt < MAX_RETRY_ATTEMPTS) {
                usleep(RETRY_DELAY_MICROSECONDS);
            }
        }
        
        return null;
    }
}
