<?php

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST');
header('Access-Control-Allow-Headers: Content-Type, Accept');

function sendResponse($statusCode, $success, $message, $data = null, $query = null, $totalResults = null, $appliedLimit = null) {
    http_response_code($statusCode);
    $response = [
        'ok' => $success,
        'dev' => '@OnyxApiTeam',
        'message' => $message
    ];
    if ($query !== null) {
        $response['query'] = $query;
    }
    if ($totalResults !== null) {
        $response['total_results'] = $totalResults;
    }
    if ($appliedLimit !== null) {
        $response['applied_limit'] = $appliedLimit;
    }
    if ($data !== null) {
        $response['data'] = $data;
    }
    echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    exit();
}

function sanitizeValue($value) {
    if ($value === null || $value === '' || $value === 0) {
        return null;
    }
    return is_string($value) ? trim($value) : $value;
}

function formatPrice($price) {
    $price = floatval($price);
    if (!$price || $price <= 0) {
        return null;
    }
    
    return [
        'amount_rial' => $price,
        'amount_toman' => floor($price / 10),
        'formatted_rial' => number_format($price, 0, '', ',') . ' ریال',
        'formatted_toman' => number_format(floor($price / 10), 0, '', ',') . ' تومان'
    ];
}

function validateImageFilename($filename) {
    if (!$filename) return false;

    if (!preg_match('/^[a-zA-Z0-9_\-]+\.(jpg|jpeg|png|webp)$/i', $filename)) {
        return false;
    }
    
    if (strpos($filename, '..') !== false || strpos($filename, '/') !== false) {
        return false;
    }
    
    return true;
}

function handleImageProxy() {
    $filename = $_GET['media'] ?? '';
    
    if (!validateImageFilename($filename)) {
        sendResponse(400, false, 'Invalid image filename provided');
        return;
    }

    $imageUrl = "https://dkstatics-public.digikala.com/digikala-products/" . urlencode($filename);
    
    $ch = curl_init($imageUrl);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_TIMEOUT => 30,
        CURLOPT_CONNECTTIMEOUT => 10,
        CURLOPT_USERAGENT => 'Mozilla/5.0 (Mobile Price API/1.0)',
        CURLOPT_HEADER => true,
        CURLOPT_SSL_VERIFYPEER => true,
        CURLOPT_MAXREDIRS => 3
    ]);

    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $headerSize = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
    $curlError = curl_error($ch);

    if ($curlError) {
        curl_close($ch);
        error_log('[Image Proxy cURL Error] ' . $curlError);
        sendResponse(503, false, 'Image service temporarily unavailable');
        return;
    }

    if ($httpCode !== 200) {
        curl_close($ch);
        sendResponse(404, false, 'Image not found');
        return;
    }

    $headers = substr($response, 0, $headerSize);
    $body = substr($response, $headerSize);

    if (preg_match('/Content-Type:\s*([^\r\n]+)/i', $headers, $matches)) {
        $contentType = trim($matches[1]);

        if (in_array($contentType, ['image/jpeg', 'image/png', 'image/webp'])) {
            header('Content-Type: ' . $contentType);
        }
    }

    curl_close($ch);
    
    header('Cache-Control: public, max-age=86400'); // 24 hours
    header('Expires: ' . gmdate('D, d M Y H:i:s', time() + 86400) . ' GMT');
    
    echo $body;
    exit();
}

function validateSearchQuery($query) {
    if (!$query || strlen(trim($query)) < 2) {
        return false;
    }
    
    $query = preg_replace('/[<>"\']/', '', $query);
    
    return trim($query);
}

function handleMainRequest() {

    if (isset($_GET['media'])) {
        handleImageProxy();
        return;
    }

    if ($_SERVER['REQUEST_METHOD'] !== 'GET' && $_SERVER['REQUEST_METHOD'] !== 'POST') {
        sendResponse(405, false, 'Method not allowed. Only GET and POST requests are supported.');
        return;
    }

    try {
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $input = json_decode(file_get_contents('php://input'), true);
            $searchQuery = $input['q'] ?? null;
            $limitQuery = $input['limit'] ?? null;
        } else {
            $searchQuery = $_GET['q'] ?? null;
            $limitQuery = $_GET['limit'] ?? null;
        }

        $validatedQuery = validateSearchQuery($searchQuery);
        if (!$validatedQuery) {
            sendResponse(400, false, 'Please provide a valid search query in the "q" parameter (minimum 2 characters).');
            return;
        }

        $limit = null;
        if ($limitQuery !== null) {
            $limit = intval($limitQuery);
            if ($limit < 1 || $limit > 100) {
                sendResponse(400, false, 'Limit parameter must be between 1 and 100.');
                return;
            }
        }

        $apiUrl = 'https://api.digikala.com/v1/categories/mobile-phone/search/?' . http_build_query([
            'q' => $validatedQuery,
            'sort' => 22,
            'page' => 1,
            'per_page' => $limit ?: 50
        ]);

        $ch = curl_init($apiUrl);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_TIMEOUT => 30,
            CURLOPT_CONNECTTIMEOUT => 10,
            CURLOPT_USERAGENT => 'Mozilla/5.0 (Mobile Price API/1.0)',
            CURLOPT_HTTPHEADER => [
                'Accept: application/json',
                'Accept-Language: fa-IR,fa;q=0.9'
            ],
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_MAXREDIRS => 3
        ]);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $curlError = curl_error($ch);

        if ($curlError) {
            curl_close($ch);
            error_log('[API cURL Error] ' . $curlError);
            sendResponse(503, false, 'External service temporarily unavailable. Please try again later.');
            return;
        }

        if ($httpCode !== 200) {
            curl_close($ch);
            error_log("[API Error] Digikala API returned HTTP $httpCode");
            
            if ($httpCode === 404) {
                sendResponse(404, false, 'No products found for the given search query.');
            } else {
                sendResponse(503, false, 'External service error. Please try again later.');
            }
            return;
        }

        curl_close($ch);

        $apiData = json_decode($response, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            error_log('[JSON Error] ' . json_last_error_msg());
            sendResponse(502, false, 'Invalid response from external service.');
            return;
        }

        $products = $apiData['data']['products'] ?? [];

        if ($limit && count($products) > $limit) {
            $products = array_slice($products, 0, $limit);
        }

        if (empty($products)) {
            sendResponse(200, true, 'Search completed successfully, but no products were found.', [], $validatedQuery, 0);
            return;
        }

        $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
        $baseUrl = $protocol . '://' . $_SERVER['HTTP_HOST'] . $_SERVER['SCRIPT_NAME'];

        $processedProducts = [];
        foreach ($products as $product) {
            $imageUrl = $product['images']['main']['url'][0] ?? '';
            $imageFilename = $imageUrl ? basename(parse_url($imageUrl, PHP_URL_PATH)) : '';
            $proxyImageUrl = ($imageFilename && validateImageFilename($imageFilename)) 
                ? $baseUrl . '?media=' . urlencode($imageFilename) 
                : null;

            $processedProducts[] = [
                'id' => sanitizeValue($product['id'] ?? null),
                'title_persian' => sanitizeValue($product['title_fa'] ?? null),
                'title_english' => sanitizeValue($product['title_en'] ?? null),
                'brand' => sanitizeValue($product['data_layer']['brand'] ?? $product['brand']['title_fa'] ?? null),
                'price' => formatPrice($product['default_variant']['price']['rrp_price'] ?? 0),
                'image_url' => $proxyImageUrl
            ];
        }

        sendResponse(200, true, 'Products retrieved successfully.', $processedProducts, $validatedQuery, count($processedProducts), $limit);

    } catch (Exception $e) {
        error_log('[Exception] ' . $e->getMessage() . ' in ' . $e->getFile() . ' on line ' . $e->getLine());
        sendResponse(500, false, 'An internal server error occurred. Please try again later.');
    } catch (Error $e) {
        error_log('[Fatal Error] ' . $e->getMessage() . ' in ' . $e->getFile() . ' on line ' . $e->getLine());
        sendResponse(500, false, 'A critical error occurred. Please contact support if the problem persists.');
    }
}

handleMainRequest();