<?php
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, PUT, PATCH, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, X-Requested-With, X-Csrf-Token');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(204);
    exit;
}

$allowedMethods = ['POST', 'PUT', 'PATCH'];
if (!in_array($_SERVER['REQUEST_METHOD'], $allowedMethods)) {
    http_response_code(405);
    echo json_encode([
        "ok" => false,
        "dev" => "@OnyxApiTeam",
        "error" => "Method not allowed. Use POST, PUT or PATCH."
    ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    exit;
}

$fileKey = null;
foreach ($_FILES as $key => $file) {
    if (
        isset($file['tmp_name']) && is_uploaded_file($file['tmp_name']) &&
        isset($file['error']) && $file['error'] === UPLOAD_ERR_OK
    ) {
        $fileKey = $key;
        break;
    }
}

if (!$fileKey) {
    http_response_code(400);
    echo json_encode([
        "ok" => false,
        "dev" => "@OnyxApiTeam",
        "error" => "No uploaded file found under any key."
    ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    exit;
}

$tmpPath = $_FILES[$fileKey]['tmp_name'];
$realName = basename($_FILES[$fileKey]['name']);
$filename = 'file_' . time() . '_' . preg_replace('/[^a-zA-Z0-9_\.-]/', '_', $realName);

$mime = mime_content_type($tmpPath);
if (substr($mime, 0, 6) !== 'image/') {
    http_response_code(415);
    echo json_encode([
        "ok" => false,
        "dev" => "@OnyxApiTeam",
        "error" => "Only image files are allowed. Detected MIME: $mime"
    ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    exit;
}

function getCsrfToken($cookieJar) {
    $ch = curl_init("https://www.eboo.ir/تبدیل-عکس-به-متن-فارسی");
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_COOKIEJAR => $cookieJar,
        CURLOPT_COOKIEFILE => $cookieJar,
        CURLOPT_USERAGENT => 'Mozilla/5.0 (compatible; CompanyOCR/1.0)'
    ]);
    $html = curl_exec($ch);
    curl_close($ch);
    if (preg_match('/name="csrf-token" content="([^"]+)"/', $html, $matches)) {
        return $matches[1];
    }
    return '';
}

function uploadFile($tmpPath, $mime, $filename, $csrf, $cookieJar) {
    $ch = curl_init("https://www.eboo.ir/Controllers/AjaxController/UploadFileToServer");
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_HTTPHEADER => [
            "x-csrf-token: $csrf",
            "x-requested-with: XMLHttpRequest"
        ],
        CURLOPT_POSTFIELDS => [
            "UploadedFileHandle" => new CURLFile($tmpPath, $mime, $filename)
        ],
        CURLOPT_COOKIEFILE => $cookieJar,
        CURLOPT_COOKIEJAR => $cookieJar,
    ]);
    $resp = curl_exec($ch);
    curl_close($ch);
    return $resp ?: '';
}

function checkFile($serverFilename, $filename, $uniqID, $csrf, $cookieJar) {
    $postData = http_build_query([
        "filename" => $serverFilename,
        "realfilename" => $filename,
        "ConvertMode" => "",
        "UniqID" => $uniqID,
        "FreeConvert" => "false"
    ]);
    $ch = curl_init("https://www.eboo.ir/Controllers/AjaxController/CheckUploadedFile");
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_HTTPHEADER => [
            "x-csrf-token" => $csrf,
            "x-requested-with" => "XMLHttpRequest"
        ],
        CURLOPT_POSTFIELDS => $postData,
        CURLOPT_COOKIEFILE => $cookieJar,
        CURLOPT_COOKIEJAR => $cookieJar,
    ]);
    curl_exec($ch);
    curl_close($ch);
}

function getOCRResult($serverFilename, $uniqID, $csrf, $cookieJar, $tries = 5, $delay = 3) {
    $url = "https://www.eboo.ir/Controllers/AjaxController/TestConvert";
    for ($i = 0; $i < $tries; $i++) {
        sleep($delay);
        $postData = http_build_query([
            "filename" => $serverFilename,
            "pagetotest" => "",
            "UniqID" => $uniqID,
            "OutputType" => "onlyTextoutput"
        ]);
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => [
                "x-csrf-token" => $csrf,
                "x-requested-with" => "XMLHttpRequest"
            ],
            CURLOPT_POSTFIELDS => $postData,
            CURLOPT_COOKIEFILE => $cookieJar,
            CURLOPT_COOKIEJAR => $cookieJar,
        ]);
        $response = curl_exec($ch);
        curl_close($ch);
        $json = json_decode($response, true);
        if ($json && isset($json['Content']) && $json['Status'] === 'PrintOutput') {
            return $json['Content'];
        }
    }
    return null;
}

$cookieJar = tempnam(sys_get_temp_dir(), 'cookie_');
$csrf = getCsrfToken($cookieJar);

if (!$csrf) {
    http_response_code(500);
    unlink($cookieJar);
    echo json_encode([
        "ok" => false,
        "dev" => "@OnyxApiTeam",
        "error" => "Failed to retrieve CSRF token."
    ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    exit;
}

$uploadResp = uploadFile($tmpPath, $mime, $filename, $csrf, $cookieJar);

if (!preg_match('/^[\w\-.]+\.(jpg|png|jpeg)$/i', trim($uploadResp))) {
    unlink($cookieJar);
    http_response_code(500);
    echo json_encode([
        "ok" => false,
        "dev" => "@OnyxApiTeam",
        "error" => "Upload failed.",
        "debug" => $uploadResp
    ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    exit;
}

$serverFilename = trim($uploadResp);
$uniqID = substr(md5($serverFilename), 0, 16);

checkFile($serverFilename, $filename, $uniqID, $csrf, $cookieJar);

$ocrResult = getOCRResult($serverFilename, $uniqID, $csrf, $cookieJar);

unlink($cookieJar);

echo json_encode([
    "ok" => (bool)$ocrResult,
    "dev" => "@OnyxApiTeam",
    $ocrResult ? "result" : "error" => $ocrResult ?: "OCR result not found."
], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
