<?php
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST');

class EbooOcr
{
    private string $cookieJar;
    private string $tempFile;
    private ?string $csrfToken = null;

    public function __construct()
    {
        $this->cookieJar = tempnam(sys_get_temp_dir(), 'cookie_');
    }

    public function run(): void
    {
        $url = $this->getInputUrl();
        if (!$url) {
            $this->response(false, 'Parameter \'url\' is required.');
            return;
        }

        $this->tempFile = tempnam(sys_get_temp_dir(), 'img_');
        if (!$this->downloadFile($url, $this->tempFile)) {
            $this->cleanup();
            $this->response(false, 'Failed to download image from URL.');
            return;
        }

        $this->csrfToken = $this->fetchCsrfToken();
        if (!$this->csrfToken) {
            $this->cleanup();
            $this->response(false, 'Failed to fetch CSRF token.');
            return;
        }

        $filename = 'file_' . time() . '.jpg';
        $serverFilename = $this->uploadFile($filename);
        if (!$serverFilename) {
            $this->cleanup();
            $this->response(false, 'Upload failed.');
            return;
        }

        $uniqID = substr(md5($serverFilename), 0, 16);
        $this->checkUploadedFile($serverFilename, $filename, $uniqID);

        $ocrResult = $this->getOcrResult($serverFilename, $uniqID);
        $this->cleanup();

        if ($ocrResult) {
            $this->response(true, null, $ocrResult);
        } else {
            $this->response(false, 'OCR result not found.');
        }
    }

    private function getInputUrl(): ?string
    {
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $input = json_decode(file_get_contents('php://input'), true);
            return $input['url'] ?? null;
        }
        return $_GET['url'] ?? null;
    }

    private function downloadFile(string $url, string $saveTo): bool
    {
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64)'
        ]);
        $data = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($httpCode === 200 && $data !== false) {
            file_put_contents($saveTo, $data);
            return true;
        }
        return false;
    }

    private function fetchCsrfToken(): ?string
    {
        $ch = curl_init("https://www.eboo.ir/تبدیل-عکس-به-متن-فارسی");
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_COOKIEJAR => $this->cookieJar,
            CURLOPT_COOKIEFILE => $this->cookieJar,
            CURLOPT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64)'
        ]);
        $html = curl_exec($ch);
        curl_close($ch);

        if (preg_match('/name="csrf-token" content="([^"]+)"/', $html, $matches)) {
            return $matches[1];
        }
        return null;
    }

    private function uploadFile(string $filename): ?string
    {
        $ch = curl_init("https://www.eboo.ir/Controllers/AjaxController/UploadFileToServer");
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => [
                "x-csrf-token: {$this->csrfToken}",
                "x-requested-with: XMLHttpRequest"
            ],
            CURLOPT_POSTFIELDS => [
                "UploadedFileHandle" => new CURLFile($this->tempFile, mime_content_type($this->tempFile), $filename)
            ],
            CURLOPT_COOKIEFILE => $this->cookieJar,
            CURLOPT_COOKIEJAR => $this->cookieJar
        ]);
        $response = curl_exec($ch);
        curl_close($ch);

        $response = trim($response);
        if (preg_match('/^[\w\-.]+\.jpg$/', $response)) {
            return $response;
        }
        return null;
    }

    private function checkUploadedFile(string $serverFilename, string $realFilename, string $uniqID): void
    {
        $ch = curl_init("https://www.eboo.ir/Controllers/AjaxController/CheckUploadedFile");
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => [
                "x-csrf-token: {$this->csrfToken}",
                "x-requested-with: XMLHttpRequest"
            ],
            CURLOPT_POSTFIELDS => http_build_query([
                "filename" => $serverFilename,
                "realfilename" => $realFilename,
                "ConvertMode" => "",
                "UniqID" => $uniqID,
                "FreeConvert" => "false"
            ]),
            CURLOPT_COOKIEFILE => $this->cookieJar,
            CURLOPT_COOKIEJAR => $this->cookieJar
        ]);
        curl_exec($ch);
        curl_close($ch);
    }

    private function getOcrResult(string $serverFilename, string $uniqID): ?string
    {
        $tries = 3;
        while ($tries-- > 0) {
            sleep(3);
            $ch = curl_init("https://www.eboo.ir/Controllers/AjaxController/TestConvert");
            curl_setopt_array($ch, [
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_POST => true,
                CURLOPT_HTTPHEADER => [
                    "x-csrf-token: {$this->csrfToken}",
                    "x-requested-with: XMLHttpRequest"
                ],
                CURLOPT_POSTFIELDS => http_build_query([
                    "filename" => $serverFilename,
                    "pagetotest" => "",
                    "UniqID" => $uniqID,
                    "OutputType" => "onlyTextoutput"
                ]),
                CURLOPT_COOKIEFILE => $this->cookieJar,
                CURLOPT_COOKIEJAR => $this->cookieJar
            ]);
            $response = curl_exec($ch);
            curl_close($ch);

            $json = json_decode($response, true);
            if (isset($json['Content']) && ($json['Status'] ?? '') === 'PrintOutput') {
                return $json['Content'];
            }
        }
        return null;
    }

    private function cleanup(): void
    {
        if (file_exists($this->tempFile)) {
            unlink($this->tempFile);
        }
        if (file_exists($this->cookieJar)) {
            unlink($this->cookieJar);
        }
    }

    private function response(bool $ok, ?string $error = null, ?string $result = null): void
    {
        $output = ['ok' => $ok, 'dev' => '@OnyxApiTeam'];
        if ($ok) {
            $output['result'] = $result;
        } else {
            $output['error'] = $error;
        }
        echo json_encode($output, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    }
}

$ocr = new EbooOcr();
$ocr->run();
