<?php
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST');

class PrayerTimesAPI {
    
    public function __construct() {
        $this->processRequest();
    }
    
    private function processRequest() {
        try {
            $query = $this->getSearchQuery();
            $location = $this->findLocation($query);
            $prayerTimes = $this->getPrayerTimes($location);
            
            $this->sendResponse([
                'ok' => true,
                'dev' => '@OnyxApiTeam',
                'result' => [
                    'location' => $location,
                    'prayer_times' => $prayerTimes
                ]
            ]);
            
        } catch (Exception $e) {
            $this->sendError($e->getMessage());
        }
    }
    
    private function getSearchQuery() {
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $input = json_decode(file_get_contents('php://input'), true);
            $q = $input['q'] ?? '';
        } else {
            $q = $_GET['q'] ?? '';
        }
        if (empty($q)) {
            throw new Exception('پارامتر q الزامی است');
        }
        return trim($q);
    }
    
    private function findLocation($query) {
        $response = $this->makeHttpRequest(
            'https://www.bahesab.ir/sdn/zsearch/',
            $this->getSearchHeaders(),
            ['string_o' => json_encode(['a' => $query], JSON_UNESCAPED_UNICODE)]
        );
        
        $data = json_decode($response, true);
        
        if (empty($data['c'])) {
            throw new Exception('مکانی با این نام یافت نشد');
        }
        
        return $this->parseLocationData($data['v']);
    }
    
    private function parseLocationData($html) {
        $dom = new DOMDocument();
        @$dom->loadHTML(mb_convert_encoding($html, 'HTML-ENTITIES', 'UTF-8'));
        
        $firstResult = $dom->getElementsByTagName('div')->item(0);
        if (!$firstResult) {
            throw new Exception('خطا در پردازش اطلاعات مکان');
        }
        
        $location = [
            'coordinates' => [
                'latitude' => $firstResult->getAttribute('lat'),
                'longitude' => $firstResult->getAttribute('lng')
            ],
            'address_parts' => []
        ];
        
        foreach ($firstResult->getElementsByTagName('span') as $span) {
            $text = trim($span->textContent);
            $cleanText = trim(str_replace(',', '', $text));
            if (!empty($cleanText)) {
                $location['address_parts'][] = $cleanText;
            }
        }
        
        $location['city'] = $location['address_parts'][0] ?? '';
        $location['province'] = end($location['address_parts']) ?? '';
        $location['full_address'] = implode(', ', $location['address_parts']);
        
        return $location;
    }
    
    private function getPrayerTimes($location) {
        $response = $this->makeHttpRequest(
            'https://www.bahesab.ir/mdn/time/azanv1/',
            $this->getPrayerTimeHeaders(),
            ['string_o' => json_encode([
                'v' => [
                    'l1' => $location['coordinates']['latitude'],
                    'l2' => $location['coordinates']['longitude'],
                    'day' => 0,
                    'n' => $location['full_address'],
                    'Flag' => date('Y/n/j')
                ]
            ], JSON_UNESCAPED_UNICODE)]
        );
        
        return json_decode($response, true);
    }
    
    private function makeHttpRequest($url, $headers, $postData) {
        $ch = curl_init();
        
        $options = [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_ENCODING => 'gzip, deflate'
        ];
        
        if (!empty($postData)) {
            $options[CURLOPT_POST] = true;
            $options[CURLOPT_POSTFIELDS] = http_build_query($postData);
        }
        
        curl_setopt_array($ch, $options);
        
        $response = curl_exec($ch);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            throw new Exception('خطا در ارتباط با سرور');
        }
        
        return $response;
    }
    
    private function getSearchHeaders() {
        return [
            'authority: www.bahesab.ir',
            'accept: */*',
            'accept-language: fa-IR,fa;q=0.9,en-US;q=0.8,en;q=0.7',
            'content-type: application/x-www-form-urlencoded',
            'origin: https://www.bahesab.ir',
            'referer: https://www.bahesab.ir/time/mashhad/',
            'user-agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/124.0.0.0 Safari/537.36'
        ];
    }
    
    private function getPrayerTimeHeaders() {
        return [
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/124.0.0.0 Safari/537.36',
            'Content-Type: application/x-www-form-urlencoded',
            'origin: https://www.bahesab.ir',
            'referer: https://www.bahesab.ir/time/mashhad/'
        ];
    }
    
    private function sendResponse($data) {
        echo json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
        exit;
    }
    
    private function sendError($message) {
        $this->sendResponse([
            'ok' => false,
            'dev' => '@OnyxApiTeam',
            'result' => [
                'error' => $message
            ]
        ]);
    }
}

new PrayerTimesAPI();