<?php
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type, Authorization");
header("Content-Type: application/json; charset=utf-8");

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

$dev = "@OnyxApiTeam";
$url = $_GET['url'] ?? $_POST['url'] ?? null;

if (!$url) {
    sendJsonResponse(false, "Missing 'url' parameter");
    exit;
}

$cookieFile = tempnam(sys_get_temp_dir(), 'cookie_');

$html = curlGet("https://sclouddownloader.net/", $cookieFile);
$csrfToken = extractCsrfToken($cookieFile);

if (!$csrfToken) {
    unlink($cookieFile);
    sendJsonResponse(false, "Failed to extract CSRF token");
    exit;
}

$postData = http_build_query([
    'csrfmiddlewaretoken' => $csrfToken,
    'url' => $url
]);

$html = curlPost("https://sclouddownloader.net/download-sound-track", $postData, $cookieFile);

unlink($cookieFile);

$trackInfo = parseTrackInfo($html);

if (!$trackInfo['title'] || !$trackInfo['download_mp3']) {
    sendJsonResponse(false, "Failed to parse track info or download link");
    exit;
}

// حذف پسوند .mp3 از انتهای عنوان اگر وجود داشته باشد
if (substr($trackInfo['title'], -4) === '.mp3') {
    $trackInfo['title'] = substr($trackInfo['title'], 0, -4);
}

$mediaDir = __DIR__ . "/media/";
$baseUrl = getBaseUrl();
$mediaUrlBase = $baseUrl . "/media/";

if (!is_dir($mediaDir)) {
    mkdir($mediaDir, 0755, true);
}

$coverLocalUrl = null;
if ($trackInfo['cover']) {
    $coverLocalFilename = generateSafeFilename(basename(parse_url($trackInfo['cover'], PHP_URL_PATH)));
    $coverLocalPath = $mediaDir . $coverLocalFilename;
    if (downloadFile($trackInfo['cover'], $coverLocalPath)) {
        $coverLocalUrl = $mediaUrlBase . $coverLocalFilename;
    }
}

$downloadLocalUrl = null;
if ($trackInfo['download_mp3']) {
    $downloadLocalFilename = generateSafeFilename(basename(parse_url($trackInfo['download_mp3'], PHP_URL_PATH)));
    $downloadLocalPath = $mediaDir . $downloadLocalFilename;
    if (downloadFile($trackInfo['download_mp3'], $downloadLocalPath)) {
        $downloadLocalUrl = $mediaUrlBase . $downloadLocalFilename;
    }
}

echo json_encode([
    'ok' => true,
    'dev' => $dev,
    'result' => [
        'title' => $trackInfo['title'],
        'owner' => $trackInfo['owner'],
        'cover' => $coverLocalUrl,
        'download_mp3' => $downloadLocalUrl
    ]
], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

function sendJsonResponse(bool $ok, string $errorMessage = null) {
    global $dev;
    $response = ['ok' => $ok, 'dev' => $dev];
    if (!$ok && $errorMessage) {
        $response['error'] = $errorMessage;
    }
    echo json_encode($response, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
}

function curlGet(string $url, string $cookieFile): string {
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_HEADER => false,
        CURLOPT_COOKIEJAR => $cookieFile,
        CURLOPT_COOKIEFILE => $cookieFile,
        CURLOPT_USERAGENT => "Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/138.0.0.0 Safari/537.36",
    ]);
    $response = curl_exec($ch);
    curl_close($ch);
    return $response ?: '';
}

function curlPost(string $url, string $postData, string $cookieFile): string {
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => $postData,
        CURLOPT_COOKIEFILE => $cookieFile,
        CURLOPT_HTTPHEADER => [
            "Content-Type: application/x-www-form-urlencoded",
            "Referer: https://sclouddownloader.net/",
            "User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/138.0.0.0 Safari/537.36"
        ],
    ]);
    $response = curl_exec($ch);
    curl_close($ch);
    return $response ?: '';
}

function extractCsrfToken(string $cookieFile): ?string {
    $content = file_get_contents($cookieFile);
    if (preg_match('/csrftoken\s+([^\s]+)/', $content, $matches)) {
        return $matches[1];
    }
    return null;
}

function parseTrackInfo(string $html): array {
    libxml_use_internal_errors(true);
    $doc = new DOMDocument();
    $doc->loadHTML('<?xml encoding="utf-8" ?>' . $html);
    $xpath = new DOMXPath($doc);

    return [
        'title' => getNodeText($xpath, "//p[@id='trackTitle']"),
        'owner' => getNodeText($xpath, "//p[@id='trackUploader']"),
        'cover' => getNodeAttr($xpath, "//img[@id='trackThumbnail']", 'src'),
        'download_mp3' => getNodeAttr($xpath, "//a[@id='trackLink']", 'href'),
    ];
}

function getNodeText(DOMXPath $xpath, string $query): ?string {
    $nodes = $xpath->query($query);
    if ($nodes->length > 0) {
        return trim($nodes->item(0)->textContent);
    }
    return null;
}

function getNodeAttr(DOMXPath $xpath, string $query, string $attr): ?string {
    $nodes = $xpath->query($query);
    if ($nodes->length > 0) {
        return $nodes->item(0)->getAttribute($attr);
    }
    return null;
}

function generateSafeFilename(string $filename): string {
    $filename = preg_replace('/[^a-zA-Z0-9_\-\.]/', '_', $filename);
    return time() . "_" . $filename;
}

function downloadFile(string $url, string $savePath): bool {
    $fp = fopen($savePath, 'w+');
    if (!$fp) {
        return false;
    }

    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_FILE => $fp,
        CURLOPT_TIMEOUT => 60,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_USERAGENT => "Mozilla/5.0",
    ]);
    $result = curl_exec($ch);
    curl_close($ch);
    fclose($fp);

    return $result !== false;
}

function getBaseUrl(): string {
    $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? "https" : "http";
    $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
    $scriptDir = rtrim(dirname($_SERVER['SCRIPT_NAME']), '/\\');
    return $scheme . "://" . $host . $scriptDir;
}
