<?php
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST');
header('Content-Type: application/json; charset=utf-8');

final class ImagePromptEnhancer
{
    private $apiEndpoint = 'https://hoshi-app.ir/api/chat-gpt.php?text=';
    private $supportedStyles = [
        "realistic", "anime", "cyberpunk", "fantasy", "photographic",
        "oil painting", "pencil sketch", "3d render", "pixel art",
        "sci-fi", "dark", "minimalist"
    ];

    private $prompt;
    private $style;
    private const DEV_HANDLE = '@OnyxApiTeam';

    public function __construct(?string $prompt, ?string $style)
    {
        $this->prompt = trim((string) $prompt);
        $this->style = $style ? strtolower(trim($style)) : null;
    }

    public function process(): void
    {
        if (!$this->prompt) {
            $this->sendError(400, 'Required parameter "prompt" is missing.');
            return;
        }

        $styleInstruction = $this->getStyleInstruction();

        $finalPrompt = $this->enhancePrompt($this->prompt, $styleInstruction);
        if (!$finalPrompt) {
            $finalPrompt = $this->prompt;
        }

        $this->generateAndOutputImage($finalPrompt);
    }

    private function getStyleInstruction(): string
    {
        if ($this->style && in_array($this->style, $this->supportedStyles, true)) {
            return "The user wants the image in this style: {$this->style}.";
        }
        return '';
    }

    private function enhancePrompt(string $userPrompt, string $styleNote): ?string
    {
        $instruction = <<<TEXT
You are an expert image prompt enhancer.
- Translate Persian or non-English prompts into fluent and vivid English.
- Add strong visual descriptions, including colors, textures, and emotions.
- Ensure ethical and inclusive content, avoiding NSFW or offensive material.
- Provide concise yet detailed prompts suitable for AI image generation.
$styleNote
User prompt:
"$userPrompt"
TEXT;

        $url = $this->apiEndpoint . urlencode($instruction);
        $response = @file_get_contents($url);

        if ($response === false) return null;

        $json = json_decode($response, true);
        if (json_last_error() !== JSON_ERROR_NONE) return null;

        return $json['result'] ?? null;
    }

    private function generateAndOutputImage(string $prompt): void
    {
        $imageApiUrl = 'https://text-to-image.null-coder.workers.dev/?prompt=' . urlencode($prompt);

        $ch = curl_init($imageApiUrl);
        curl_setopt_array($ch, [
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HEADER => true,
            CURLOPT_TIMEOUT => 30,
        ]);

        $response = curl_exec($ch);
        $headerSize = curl_getinfo($ch, CURLINFO_HEADER_SIZE);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $contentType = curl_getinfo($ch, CURLINFO_CONTENT_TYPE);
        curl_close($ch);

        $body = substr($response, $headerSize);

        if ($httpCode === 200 && strpos($contentType, 'image/png') !== false) {
            $this->saveImage($body, $prompt);
        } else {
            $this->sendError(500, 'Image generation failed due to an internal server error. Please try again later.');
        }
    }

    private function saveImage(string $imageData, string $prompt): void
    {
        if (!is_dir('media')) {
            mkdir('media', 0777, true);
        }

        $uniqueId = uniqid('img_', true);
        $fileName = "media/{$uniqueId}.png";

        if (file_put_contents($fileName, $imageData) !== false) {
            $this->sendSuccess($fileName, $prompt);
        } else {
            $this->sendError(500, 'Failed to save the generated image. Please try again later.');
        }
    }

    private function sendSuccess(string $filePath, string $prompt): void
    {
        $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https://' : 'http://';
        $host = $_SERVER['HTTP_HOST'] ?? 'localhost';
        $basePath = rtrim(dirname($_SERVER['SCRIPT_NAME']), '/\\');
        $basePath = $basePath === '.' ? '' : $basePath;
        $fullLink = $protocol . $host . $basePath . '/' . ltrim($filePath, '/');

        $fileSize = file_exists($filePath) ? filesize($filePath) : 0;
        $timestamp = date('Y-m-d H:i:s');

        http_response_code(200);
        echo json_encode([
            'ok' => true,
            'dev' => self::DEV_HANDLE,
            'file' => [
                'link' => $fullLink,
                'size' => $fileSize,
                'type' => 'image/png',
                'created_at' => $timestamp,
            ],
            'prompt' => stripslashes($prompt),
        ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    }

    private function sendError(int $httpCode, string $message): void
    {
        http_response_code($httpCode);
        echo json_encode([
            'ok' => false,
            'dev' => self::DEV_HANDLE,
            'error' => $message,
        ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    }
}

function getRequestParam(string $key): ?string
{
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $input = json_decode(file_get_contents('php://input'), true);
        return $input[$key] ?? null;
    }
    return $_GET[$key] ?? null;
}

$prompt = getRequestParam('prompt');
$style = getRequestParam('style');

$enhancer = new ImagePromptEnhancer($prompt, $style);
$enhancer->process();
