<?php

header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST');
header('Cache-Control: no-store');
header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] !== 'GET' && $_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode([
        'ok' => false,
        'error' => 'Method Not Allowed'
    ], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    exit;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);
    $channelsParam = $input['channel'] ?? null;
} else {
    $channelsParam = $_GET['channel'] ?? null;
}

if (!$channelsParam) {
    http_response_code(400);
    echo json_encode([
        'ok' => false,
        'dev' => '@OnyxApiTeam',
        'error' => 'Missing channel parameter'
    ], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    exit;
}

$channels = array_filter(array_map('trim', explode(',', $channelsParam)));
if (empty($channels)) {
    http_response_code(400);
    echo json_encode([
        'ok' => false,
        'dev' => '@OnyxApiTeam',
        'error' => 'No valid channels provided'
    ], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    exit;
}

$allConfigs = [];
$validChannels = [];

foreach ($channels as $channel) {
    $normalizedUrl = normalizeChannelUrl($channel);
    if (!$normalizedUrl) {
        continue;
    }

    $html = fetchChannelPage($normalizedUrl);
    if ($html === false) {
        continue;
    }

    $configs = extractV2rayConfigs($html);
    if (empty($configs)) {
        continue;
    }

    $validChannels[] = extractChannelHandle($channel);
    $allConfigs = array_merge($allConfigs, $configs);
}

if (!empty($allConfigs)) {
    echo json_encode([[
        'ok' => true,
        'dev' => '@OnyxApiTeam',
        'channel' => implode(',', $validChannels),
        'configs' => array_values(array_unique($allConfigs))
    ]], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
} else {
    echo json_encode([[
        'ok' => false,
        'dev' => '@OnyxApiTeam',
        'channel' => implode(',', array_map('extractChannelHandle', $channels)),
        'error' => 'No V2Ray configs found in channel'
    ]], JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
}


function fetchChannelPage($url) {
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
    curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0');
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    $html = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    if ($httpCode !== 200 || $html === false) return false;
    return html_entity_decode($html);
}

function extractChannelHandle($input) {
    $input = trim($input);
    if (strpos($input, '@') === 0) return $input;
    if (strpos($input, 't.me/') !== false) {
        $parts = explode('/', $input);
        $username = end($parts) ?: prev($parts);
        return '@' . $username;
    }
    return '@' . $input;
}

function extractV2rayConfigs($html) {
    $patterns = [
        '/vmess:\/\/[a-zA-Z0-9+\/=._-]+/i',
        '/vless:\/\/[a-zA-Z0-9@:%._\+~#=;\/?-]+/i',
        '/trojan:\/\/[a-zA-Z0-9@:%._\+~#=;\/?-]+/i',
        '/ss:\/\/[a-zA-Z0-9+\/=._-]+/i'
    ];
    $allMatches = [];
    foreach ($patterns as $pattern) {
        preg_match_all($pattern, $html, $matches);
        $allMatches = array_merge($allMatches, $matches[0]);
    }
    return array_values(array_unique($allMatches));
}

function normalizeChannelUrl($input) {
    $input = trim($input);
    if (strpos($input, '@') === 0) return "https://t.me/s/" . substr($input, 1);
    if (strpos($input, 'https://t.me/') === 0) {
        $urlParts = explode('/', str_replace('https://t.me/', '', $input));
        $username = $urlParts[0];
        if ($username === 's' && isset($urlParts[1])) $username = $urlParts[1];
        return "https://t.me/s/$username";
    }
    if (strpos($input, 't.me/') === 0) {
        $urlParts = explode('/', str_replace('t.me/', '', $input));
        $username = $urlParts[0];
        if ($username === 's' && isset($urlParts[1])) $username = $urlParts[1];
        return "https://t.me/s/$username";
    }
    if ($input && strpos($input, '/') === false && strpos($input, '@') === false) {
        return "https://t.me/s/$input";
    }
    return null;
}
