<?php

class AIManager {
    private $config;
    private $pdo;
    private $keyIndex = 0;

    public function __construct(array $config) {
        $this->config = $config;
        $db = $config['db'];
        $dsn = "mysql:host={$db['host']};dbname={$db['name']};charset=utf8mb4";
        $this->pdo = new PDO($dsn, $db['user'], $db['pass'], [
            PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION
        ]);
        $this->pdo->exec("
            CREATE TABLE IF NOT EXISTS chat_history (
                id INT AUTO_INCREMENT PRIMARY KEY,
                user_id VARCHAR(255) NOT NULL,
                role ENUM('user','model') NOT NULL,
                content TEXT NOT NULL,
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
        ");
    }

    private function getNextApiKey(): string {
        $keys = $this->config['gemini']['api_keys'] ?? $this->config['gemini_api_keys'];
        $key = $keys[$this->keyIndex];
        $this->keyIndex = ($this->keyIndex + 1) % count($keys);
        return $key;
    }

    public function saveMessage(string $userId, string $role, string $content): void {
        $stmt = $this->pdo->prepare("INSERT INTO chat_history (user_id, role, content) VALUES (?, ?, ?)");
        $stmt->execute([$userId, $role, $content]);
        if (!isset($this->config['history_limit'])) {
            throw new Exception('history_limit must be set in config.php');
        }
        $limit = (int)$this->config['history_limit'];
        $cleanup = $this->pdo->prepare("
            DELETE FROM chat_history
            WHERE user_id = ?
            AND id NOT IN (
                SELECT id FROM (
                    SELECT id FROM chat_history
                    WHERE user_id = ?
                    ORDER BY id DESC
                    LIMIT $limit
                ) AS t
            )
        ");
        $cleanup->execute([$userId, $userId]);
    }

    private function getLastMessages(string $userId, int $limit = null): array {
        if ($limit === null) {
            if (!isset($this->config['history_limit'])) {
                throw new Exception('history_limit must be set in config.php');
            }
            $limit = (int)$this->config['history_limit'] - 1;
        }
        $stmt = $this->pdo->prepare("SELECT role, content FROM chat_history WHERE user_id = ? ORDER BY id DESC LIMIT ?");
        $stmt->bindValue(1, $userId, PDO::PARAM_STR);
        $stmt->bindValue(2, $limit, PDO::PARAM_INT);
        $stmt->execute();
        return array_reverse($stmt->fetchAll(PDO::FETCH_ASSOC));
    }

    public function generateText(string $userId, string $message) {
        $this->saveMessage($userId, 'user', $message);
    $history = $this->getLastMessages($userId);

        $messages = [
            [
                "role" => "user",
                "parts" => [["text" => $this->config['system_instruction']]]
            ]
        ];

        foreach ($history as $row) {
            $messages[] = [
                "role" => $row['role'],
                "parts" => [["text" => $row['content']]]
            ];
        }

        $messages[] = [
            "role" => "user",
            "parts" => [["text" => $message]]
        ];

        $payload = json_encode(["contents" => $messages]);
        $gemini = $this->config['gemini'] ?? [];
        $keys = $gemini['api_keys'] ?? $this->config['gemini_api_keys'];
        if (empty($gemini['model'])) {
            return ['error' => 'Gemini model must be set in config.php'];
        }
        $model = $gemini['model'];
        $endpointPattern = $gemini['endpoint_pattern'] ?? null;
        if (!$endpointPattern) {
            $endpointPattern = $this->config['gemini_endpoint'] ?? null;
        }
        if (!$endpointPattern) {
            return ['error' => 'Gemini endpoint pattern not set in config.'];
        }
        $endpoint = str_replace('{model}', $model, $endpointPattern);
        $numKeys = count($keys);
        $startIndex = $this->keyIndex;
        $tried = 0;
        $lastError = null;
        while ($tried < $numKeys) {
            $apiKey = $this->getNextApiKey();
            $ch = curl_init($endpoint . "?key=" . $apiKey);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_HTTPHEADER, ["Content-Type: application/json"]);
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $payload);
            $response = curl_exec($ch);
            curl_close($ch);

            $data = @json_decode($response, true);
            $output = $data['candidates'][0]['content']['parts'][0]['text'] ?? null;
            if ($output) {
                $this->saveMessage($userId, 'model', $output);
                return $output;
            }
            $tried++;
        }
        return ['error' => 'No response received from the server. Please try again later.'];
    }
}
